<?php

namespace App\Http\Controllers;

use App\Http\Requests\ApiStudentPostRequest;
use App\Http\Requests\ApiStudentUploadRequest;
use App\Models\Branch;
use App\Models\Certificate;
use App\Models\Course;
use App\Models\Student;
use Illuminate\Http\Request;

class ApiStudentController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $data = [
            'branches' => Branch::select('branch_id','branch_name')->get(),
            'courses'  => Course::select('course_id','course_name')->get()
        ];

        return $this->JsonOutPut(true,"Fetched student branch and course list", $data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(ApiStudentPostRequest $request)
    {  
        $student = Student::create([
            'student_regno'     => $request->student_admission_no,
            'student_name'      => $request->student_name,
            'student_dob'       => $request->student_dob,
            'student_doj'       => $request->student_doj,
            'student_branch_id' => decrypt($request->student_branch),
            'student_course_id' => decrypt($request->student_course),
            'student_address'   => $request->student_addr
        ]);

        $updateData = Student::find($student->student_id);

        if($request->hasFile('student_aadhar')){
            $extension = $request->file('student_aadhar')->getClientOriginalExtension();
            $file_name = "aadhar_".$student->student_regno.".".$extension;
            $file      = $request->file('student_aadhar')->storeAs('images',$file_name);

            $updateData->student_aadhar_file = $file;
            $updateData->save();
        }

        if($request->hasFile('student_sslc')){
            $extension = $request->file('student_sslc')->getClientOriginalExtension();
            $file_name = "sslc_".$student->student_regno.".".$extension;
            $file      = $request->file('student_sslc')->storeAs('images',$file_name);

            $updateData->student_sslc_file = $file;
            $updateData->save();
        }

        if($request->hasFile('student_others')){
            $extension = $request->file('student_others')->getClientOriginalExtension();
            $file_name = "others_".$student->student_regno.".".$extension;
            $file      = $request->file('student_others')->storeAs('images',$file_name);

            $updateData->student_other_file = $file;
            $updateData->save();
        }

        return $this->JsonOutPut(true,"New Admission Created");
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $limit= ($request->has('limit') && $request->limit > 0) ? $request->limit : 10;
        $page = $request->page ?? 1;
        $search_key = $request->term ?? null;
        
        if($request->has('sortable')){
            $studentOrder = $this->DataTableOrderBy($request->sortable);
            $student = Student::orderByRaw($studentOrder);
        }
        else{
            $student = Student::orderBy('created_at','desc');
        }
        
        if($search_key != null && $request->has('fields')){
            $collection = collect($request->fields); 
            $relations  = ['branch','course'];      
            
            $studentWhere  = $this->DataTableWhere($request->fields,$relations, $search_key);

            $student->whereRaw($studentWhere);

            if($collection->contains('branch')){
                $student->orWhereHas('branch',function($query) use ($search_key){
                    $query->where('branch_name','LIKE','%'.$search_key.'%');
                });
            }
        }
        // $sql   = $user->toSql();
        $students = $student->paginate($limit,['*'],'students',$page);
        

        $data = [
           'students'  => $students->toArray()['data'],
           'total'     => $students->total(),
           'index'     => $students->firstItem(),
        //    'sql'       => $sql
        ];
        return $this->JsonOutPut(true,'Fetched Branches list',$data);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(ApiStudentPostRequest $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    public function searchStudents(Request $request){
        $branch_id = auth()->user()->branch_id;
        if($branch_id > 1)
            $students = Student::select('student_id','student_name','student_regno','student_branch_id','student_course_id')
                        ->where('student_branch_id',$branch_id)
                        ->where('student_name','like',$request->search_key."%")
                        ->get();
        else    
            $students = Student::select('student_id','student_name','student_regno','student_branch_id','student_course_id')
                        ->where('student_name','like',$request->search_key."%")
                        ->get();
        
        $data = [
            'students'=> $students
        ];
        return $this->JsonOutPut(true,"Fetched Students list", $data);
    }


    public function uploadCertificate(ApiStudentUploadRequest $request){
        
        if($request->hasFile('upload_certificate')){
            $certid    = $request->upload_admission;
            $extension = $request->file('upload_certificate')->getClientOriginalExtension();
            $file_name = "certificate_".$certid.".".$extension;
            $file      = $request->file('upload_certificate')->storeAs('certificates',$file_name);

            Certificate::create([
                'cert_student_id' => decrypt($request->upload_student),
                'cert_course_id'  => decrypt($request->upload_course),
                'cert_image_file' => $file
            ]);
        }
        return $this->JsonOutPut(true,"Certificate Upload Finished");
    }

    public function getCertificates(Request $request){
        $limit= ($request->has('limit') && $request->limit > 0) ? $request->limit : 10;
        $page = $request->page ?? 1;
        $search_key = $request->term ?? null;
        
        if($request->has('sortable')){
            $studentOrder = $this->DataTableOrderBy($request->sortable);
            $cert = Certificate::orderByRaw($studentOrder);
        }
        else{
            $cert = Certificate::orderBy('created_at','desc');
        }
        
        if($search_key != null && $request->has('fields')){
            $collection = collect($request->fields); 
            $relations  = ['student','course'];      
            
            $certWhere  = $this->DataTableWhere($request->fields,$relations, $search_key);
            if($certWhere){
                $cert->whereRaw($certWhere);
            }    

            if($collection->contains('student')){
                $cert->orWhereHas('student',function($query) use ($search_key){
                    $query->where('student_name','LIKE','%'.$search_key.'%');
                });
            }

            if($collection->contains('course')){
                $cert->orWhereHas('course',function($query) use ($search_key){
                    $query->where('course_name','LIKE','%'.$search_key.'%');
                });
            }
        }
        $sql   = $cert->toSql();
        $certs = $cert->paginate($limit,['cert_image_file','cert_student_id','cert_course_id','created_at'],'certificates',$page);
        

        $data = [
           'certificates' => $certs->toArray()['data'],
           'total'        => $certs->total(),
           'index'        => $certs->firstItem(),
           'sql'          => $sql
        ];
        return $this->JsonOutPut(true,'Fetched Branches list',$data);
    }

    public function getStudentCertificate(Request $request){
        if(!$request->has('admission_no')){
            $data = [
                'errors' => 'Admission Number Required'
            ];
            return $this->JsonOutPut(false,'Validation Errors',$data);
        }
        else{
            $admission = $request->admission_no;
            $student = Student::with('certificate')->where('student_regno',$admission)->first();
            if($student){
                if(!$student->certificate){
                    $data = [
                        'errors' => 'Certificates not found against given admission number'
                    ];
                    return $this->JsonOutPut(false,'Not Found',$data);
                }
                else{
                    $data = [
                        'certificate' => [
                            'url' => $student->certificate->cert_image_url,
                            'mime'=> $student->certificate->cert_image_mime
                        ]
                    ];
                    return $this->JsonOutPut(true,'Fetched Student Certificate',$data);
                }    
            }
            else{
                $data = [
                    'errors' => 'Student not found against given admission number'
                ];
                return $this->JsonOutPut(false,'Not Found',$data);
            }    
        }
    }
}
