<?php

namespace App\Http\Controllers;

use App\Http\Requests\ApiCoursePostRequest;
use App\Models\Course;
use Illuminate\Http\Request;

class ApiCourseController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $data = [
            'courses' => $request->has('limit') ? Course::select('course_name','course_desc')->without('branch')->limit($request->limit)->get() : Course::all(),
        ];
        return $this->JsonOutPut(true,"Fetched list of courses",$data);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(ApiCoursePostRequest $request)
    {
        Course::create([
            'course_name' => $request->course_name,
            'course_desc' => $request->course_desc,
            'branch_id'   => decrypt($request->course_branch)
        ]);
        return $this->JsonOutPut(true,"Course Created");
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $limit= ($request->has('limit') && $request->limit > 0) ? $request->limit : 10;
        $page = $request->page ?? 1;
        $search_key = $request->term ?? null;
        
        if($request->has('sortable')){
            $courseOrder = $this->DataTableOrderBy($request->sortable);
            $course      = Course::orderByRaw($courseOrder);
        }
        else{
            $course      = Course::orderBy('created_at','desc');
        }
        
        if($search_key != null && $request->has('fields')){
            $collection = collect($request->fields); 
            $relations  = ['branch'];      
            
            $courseWhere  = $this->DataTableWhere($request->fields,$relations, $search_key);

            $course->whereRaw($courseWhere);

            if($collection->contains('branch')){
                $course->orWhereHas('branch',function($query) use ($search_key){
                    $query->where('branch_name','LIKE','%'.$search_key.'%');
                });
            }
        }
        // $sql   = $user->toSql();
        $courses = $course->paginate($limit,['*'],'courses',$page);
        

        $data = [
           'courses'   => $courses->toArray()['data'],
           'total'     => $courses->total(),
           'index'     => $courses->firstItem(),
        //    'sql'       => $sql
        ];
        return $this->JsonOutPut(true,'Fetched Courses list',$data);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $result = Course::find(decrypt($id));
        if($result){
            $data = [
                'course' => $result,
                'id'     => decrypt($id)
            ];
            return $this->JsonOutPut(true,'Fetched Course for update',$data);
        }
        else{
            $data = [
                'errors' => 'Course not found'
            ];
            return $this->JsonOutPut(false,'Failed to fetch data',$data);
        }    
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(ApiCoursePostRequest $request, $id)
    {
        $course = Course::find(decrypt($id));

        $course->course_name = $request->course_name;
        $course->course_desc = $request->course_desc;
        $course->branch_id   = decrypt($request->course_branch);

        $updated = $course->save();

        if($updated){
            $data = [
                'updated' => $updated
            ];
            return $this->JsonOutPut(true,'Course Data Updated Successfully',$data);
        }
        else{
            $data = [
                'errors' => 'Something went wrong'
            ];
            return $this->JsonOutPut(false,'Course Data Update Failed',$data);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
