<?php

namespace App\Http\Controllers;

use App\Jobs\ProcessImageUpload;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ImageUploadController extends Controller
{

    
    public function uploadImages(Request $request)
    {
        if (!$request->isMethod('post')) {
            return response()->json(['message' => 'Method Not Allowed'], 405);
        }
    
        // Validate the uploaded files
        $request->validate([
            'images' => 'required', // Ensure 'images' is present
            'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048', // Validate each file if it's an array
        ]);
    
        $uploadedFiles = $request->file('images');
    
        // Normalize to an array
        $files = is_array($uploadedFiles) ? $uploadedFiles : [$uploadedFiles];
    
        if (empty($files)) {
            \Log::error('No files uploaded.');
            return response()->json(['message' => 'No files uploaded'], 400);
        }
    
        $dispatchedJobs = [];
        foreach ($files as $file) {
            // Store the file in a temporary location
            $tempPath = $file->store('temp');
            if (!$tempPath) {
                \Log::error('Failed to store the file.');
                continue; // Skip this file if it cannot be stored
            }
    
            // Dispatch the job
            $destination = 'uploads/images';
            ProcessImageUpload::dispatch($tempPath, $destination);
            $dispatchedJobs[] = $tempPath;
        }
    
        if (count($dispatchedJobs) === 0) {
            return response()->json(['message' => 'Failed to process any images.'], 500);
        }
    
        return response()->json([
            'message' => 'Image upload jobs dispatched successfully.',
            'dispatched_files' => $dispatchedJobs,
        ]);
    }


}
