<?php
/** 
 * @package     VikBooking
 * @subpackage  core
 * @author      Alessio Gaggii - E4J s.r.l.
 * @copyright   Copyright (C) 2024 E4J s.r.l. All Rights Reserved.
 * @license     http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 * @link        https://vikwp.com
 */

// No direct access
defined('ABSPATH') or die('No script kiddies please!');

/**
 * Defines the handler for a pax field of type "spain_municipio".
 * 
 * @since 	1.17.2 (J) - 1.7.2 (WP)
 */
final class VBOCheckinPaxfieldTypeSpainMunicipio extends VBOCheckinPaxfieldType
{
	/**
	 * Renders the current pax field HTML.
	 * 
	 * @return 	string 	the HTML string to render the field.
	 */
	public function render()
	{
		// get the field unique ID
		$field_id = $this->getFieldIdAttr();

		// get the guest number
		$guest_number = $this->field->getGuestNumber();

		// get the field class attribute
		$pax_field_class = $this->getFieldClassAttr();

		// get field name attribute
		$name = $this->getFieldNameAttr();

		// get the field value attribute
		$value = $this->getFieldValueAttr();

		// compose HTML content for the field
		$field_html = '';
		$field_html .= "<select id=\"$field_id\" data-gind=\"$guest_number\" class=\"$pax_field_class\" name=\"$name\">\n";
		$field_html .= '<option></option>' . "\n";

		foreach (self::loadMunicipioCodes() as $codigo => $municipio) {
			$field_html .= '<option value="' . htmlspecialchars($codigo, ENT_QUOTES, 'UTF-8') . '"' . ($value == $codigo ? ' selected="selected"' : '') . '>' . $municipio . '</option>' . "\n";
		}

		$field_html .= '</select>';

		// append select2 JS script for rendering the field
		$field_html .= <<<HTML
<script>
	jQuery(function() {

		jQuery("#$field_id").select2({
			width: "100%",
			placeholder: "-----",
			allowClear: true
		});

	});
</script>
HTML;

		// return the necessary HTML string to display the field
		return $field_html;
	}

	/**
	 * Helper method that returns a list of "municipio codes".
	 *
	 * @return 	array
	 */
	public static function loadMunicipioCodes()
	{
		return [
			'01051' => "Agurain/Salvatierra",
			'01001' => "Alegría-Dulantzi",
			'01002' => "Amurrio",
			'01049' => "Añana",
			'01003' => "Aramaio",
			'01006' => "Armiñón",
			'01037' => "Arraia-Maeztu",
			'01008' => "Arratzua-Ubarrundia",
			'01004' => "Artziniega",
			'01009' => "Asparrena",
			'01010' => "Ayala/Aiara",
			'01011' => "Baños de Ebro/Mañueta",
			'01013' => "Barrundia",
			'01014' => "Berantevilla",
			'01016' => "Bernedo",
			'01017' => "Campezo/Kanpezu",
			'01021' => "Elburgo/Burgelu",
			'01022' => "Elciego",
			'01023' => "Elvillar/Bilar",
			'01046' => "Erriberagoitia/Ribera Alta",
			'01056' => "Harana/Valle de Arana",
			'01901' => "Iruña Oka/Iruña de Oca",
			'01027' => "Iruraiz-Gauna",
			'01019' => "Kripan",
			'01020' => "Kuartango",
			'01028' => "Labastida/Bastida",
			'01030' => "Lagrán",
			'01031' => "Laguardia",
			'01032' => "Lanciego/Lantziego",
			'01902' => "Lantarón",
			'01033' => "Lapuebla de Labarca",
			'01036' => "Laudio/Llodio",
			'01058' => "Legutio",
			'01034' => "Leza",
			'01039' => "Moreda de Álava/Moreda Araba",
			'01041' => "Navaridas",
			'01042' => "Okondo",
			'01043' => "Oyón-Oion",
			'01044' => "Peñacerrada-Urizaharra",
			'01047' => "Ribera Baja/Erriberabeitia",
			'01052' => "Samaniego",
			'01053' => "San Millán/Donemiliaga",
			'01054' => "Urkabustaiz",
			'01055' => "Valdegovía/Gaubea",
			'01057' => "Villabuena de Álava/Eskuernaga",
			'01059' => "Vitoria-Gasteiz",
			'01060' => "Yécora/Iekora",
			'01061' => "Zalduondo",
			'01062' => "Zambrana",
			'01018' => "Zigoitia",
			'01063' => "Zuia",
			'02001' => "Abengibre",
			'02002' => "Alatoz",
			'02003' => "Albacete",
			'02004' => "Albatana",
			'02005' => "Alborea",
			'02006' => "Alcadozo",
			'02007' => "Alcalá del Júcar",
			'02008' => "Alcaraz",
			'02009' => "Almansa",
			'02010' => "Alpera",
			'02011' => "Ayna",
			'02012' => "Balazote",
			'02014' => "Ballestero, El",
			'02013' => "Balsa de Ves",
			'02015' => "Barrax",
			'02016' => "Bienservida",
			'02017' => "Bogarra",
			'02018' => "Bonete",
			'02019' => "Bonillo, El",
			'02020' => "Carcelén",
			'02021' => "Casas de Juan Núñez",
			'02022' => "Casas de Lázaro",
			'02023' => "Casas de Ves",
			'02024' => "Casas-Ibáñez",
			'02025' => "Caudete",
			'02026' => "Cenizate",
			'02029' => "Chinchilla de Monte-Aragón",
			'02027' => "Corral-Rubio",
			'02028' => "Cotillas",
			'02030' => "Elche de la Sierra",
			'02031' => "Férez",
			'02032' => "Fuensanta",
			'02033' => "Fuente-Álamo",
			'02034' => "Fuentealbilla",
			'02035' => "Gineta, La",
			'02036' => "Golosalvo",
			'02037' => "Hellín",
			'02038' => "Herrera, La",
			'02039' => "Higueruela",
			'02040' => "Hoya-Gonzalo",
			'02041' => "Jorquera",
			'02042' => "Letur",
			'02043' => "Lezuza",
			'02044' => "Liétor",
			'02045' => "Madrigueras",
			'02046' => "Mahora",
			'02047' => "Masegoso",
			'02048' => "Minaya",
			'02049' => "Molinicos",
			'02050' => "Montalvos",
			'02051' => "Montealegre del Castillo",
			'02052' => "Motilleja",
			'02053' => "Munera",
			'02054' => "Navas de Jorquera",
			'02055' => "Nerpio",
			'02056' => "Ontur",
			'02057' => "Ossa de Montiel",
			'02058' => "Paterna del Madera",
			'02060' => "Peñas de San Pedro",
			'02059' => "Peñascosa",
			'02061' => "Pétrola",
			'02062' => "Povedilla",
			'02901' => "Pozo Cañada",
			'02063' => "Pozohondo",
			'02064' => "Pozo-Lorente",
			'02065' => "Pozuelo",
			'02066' => "Recueja, La",
			'02067' => "Riópar",
			'02068' => "Robledo",
			'02069' => "Roda, La",
			'02070' => "Salobre",
			'02071' => "San Pedro",
			'02072' => "Socovos",
			'02073' => "Tarazona de la Mancha",
			'02074' => "Tobarra",
			'02075' => "Valdeganga",
			'02076' => "Vianos",
			'02077' => "Villa de Ves",
			'02078' => "Villalgordo del Júcar",
			'02079' => "Villamalea",
			'02080' => "Villapalacios",
			'02081' => "Villarrobledo",
			'02082' => "Villatoya",
			'02083' => "Villavaliente",
			'02084' => "Villaverde de Guadalimar",
			'02085' => "Viveros",
			'02086' => "Yeste",
			'03002' => "Agost",
			'03003' => "Agres",
			'03004' => "Aigües",
			'03014' => "Alacant/Alicante",
			'03005' => "Albatera",
			'03006' => "Alcalalí",
			'03009' => "Alcoi/Alcoy",
			'03008' => "Alcoleja",
			'03007' => "Alcosser",
			'03010' => "Alfafara",
			'03011' => "Alfàs del Pi, l'",
			'03012' => "Algorfa",
			'03013' => "Algueña",
			'03015' => "Almoradí",
			'03016' => "Almudaina",
			'03017' => "Alqueria d'Asnar, l'",
			'03018' => "Altea",
			'03019' => "Aspe",
			'03001' => "Atzúbia, l'",
			'03020' => "Balones",
			'03021' => "Banyeres de Mariola",
			'03022' => "Benasau",
			'03023' => "Beneixama",
			'03024' => "Benejúzar",
			'03025' => "Benferri",
			'03026' => "Beniarbeig",
			'03027' => "Beniardà",
			'03028' => "Beniarrés",
			'03030' => "Benidoleig",
			'03031' => "Benidorm",
			'03032' => "Benifallim",
			'03033' => "Benifato",
			'03029' => "Benigembla",
			'03034' => "Benijófar",
			'03035' => "Benilloba",
			'03036' => "Benillup",
			'03037' => "Benimantell",
			'03038' => "Benimarfull",
			'03039' => "Benimassot",
			'03040' => "Benimeli",
			'03041' => "Benissa",
			'03043' => "Biar",
			'03044' => "Bigastro",
			'03045' => "Bolulla",
			'03046' => "Busot",
			'03049' => "Callosa de Segura",
			'03048' => "Callosa d'en Sarrià",
			'03047' => "Calp",
			'03051' => "Camp de Mirra, el/Campo de Mirra",
			'03050' => "Campello, el",
			'03052' => "Cañada",
			'03053' => "Castalla",
			'03054' => "Castell de Castells",
			'03075' => "Castell de Guadalest, el",
			'03055' => "Catral",
			'03056' => "Cocentaina",
			'03057' => "Confrides",
			'03058' => "Cox",
			'03059' => "Crevillent",
			'03061' => "Daya Nueva",
			'03062' => "Daya Vieja",
			'03063' => "Dénia",
			'03064' => "Dolores",
			'03066' => "Elda",
			'03065' => "Elx/Elche",
			'03067' => "Fageca",
			'03068' => "Famorca",
			'03069' => "Finestrat",
			'03077' => "Fondó de les Neus, el/Hondón de las Nieves",
			'03070' => "Formentera del Segura",
			'03072' => "Gaianes",
			'03071' => "Gata de Gorgos",
			'03073' => "Gorga",
			'03074' => "Granja de Rocamora",
			'03076' => "Guardamar del Segura",
			'03078' => "Hondón de los Frailes",
			'03079' => "Ibi",
			'03080' => "Jacarilla",
			'03085' => "Llíber",
			'03086' => "Millena",
			'03088' => "Monforte del Cid",
			'03089' => "Monòver/Monóvar",
			'03903' => "Montesinos, Los",
			'03091' => "Murla",
			'03092' => "Muro de Alcoy",
			'03090' => "Mutxamel",
			'03093' => "Novelda",
			'03094' => "Nucia, la",
			'03095' => "Ondara",
			'03096' => "Onil",
			'03097' => "Orba",
			'03099' => "Orihuela",
			'03084' => "Orxa, l'/Lorcha",
			'03098' => "Orxeta",
			'03100' => "Parcent",
			'03101' => "Pedreguer",
			'03102' => "Pego",
			'03103' => "Penàguila",
			'03104' => "Petrer",
			'03902' => "Pilar de la Horadada",
			'03105' => "Pinós, el/Pinoso",
			'03106' => "Planes",
			'03042' => "Poble Nou de Benitatxell, el/Benitachell",
			'03901' => "Poblets, els",
			'03107' => "Polop",
			'03060' => "Quatretondeta",
			'03109' => "Rafal",
			'03110' => "Ràfol d'Almúnia, el",
			'03111' => "Redován",
			'03112' => "Relleu",
			'03113' => "Rojales",
			'03114' => "Romana, la",
			'03115' => "Sagra",
			'03116' => "Salinas",
			'03118' => "San Fulgencio",
			'03904' => "San Isidro",
			'03120' => "San Miguel de Salinas",
			'03117' => "Sanet y Negrals",
			'03119' => "Sant Joan d'Alacant",
			'03122' => "Sant Vicent del Raspeig/San Vicente del Raspeig",
			'03121' => "Santa Pola",
			'03123' => "Sax",
			'03124' => "Sella",
			'03125' => "Senija",
			'03127' => "Tàrbena",
			'03128' => "Teulada",
			'03129' => "Tibi",
			'03130' => "Tollos",
			'03131' => "Tormos",
			'03132' => "Torre de les Maçanes, la/Torremanzanas",
			'03133' => "Torrevieja",
			'03134' => "Vall d'Alcalà, la",
			'03136' => "Vall de Gallinera, la",
			'03137' => "Vall de Laguar, la",
			'03135' => "Vall d'Ebo, la",
			'03138' => "Verger, el",
			'03139' => "Vila Joiosa, la/Villajoyosa",
			'03140' => "Villena",
			'03082' => "Xàbia/Jávea",
			'03081' => "Xaló",
			'03083' => "Xixona/Jijona",
			'04001' => "Abla",
			'04002' => "Abrucena",
			'04003' => "Adra",
			'04004' => "Albanchez",
			'04005' => "Alboloduy",
			'04006' => "Albox",
			'04007' => "Alcolea",
			'04008' => "Alcóntar",
			'04009' => "Alcudia de Monteagud",
			'04010' => "Alhabia",
			'04011' => "Alhama de Almería",
			'04012' => "Alicún",
			'04013' => "Almería",
			'04014' => "Almócita",
			'04015' => "Alsodux",
			'04016' => "Antas",
			'04017' => "Arboleas",
			'04018' => "Armuña de Almanzora",
			'04019' => "Bacares",
			'04904' => "Balanegra",
			'04020' => "Bayárcal",
			'04021' => "Bayarque",
			'04022' => "Bédar",
			'04023' => "Beires",
			'04024' => "Benahadux",
			'04026' => "Benitagla",
			'04027' => "Benizalón",
			'04028' => "Bentarique",
			'04029' => "Berja",
			'04030' => "Canjáyar",
			'04031' => "Cantoria",
			'04032' => "Carboneras",
			'04033' => "Castro de Filabres",
			'04036' => "Chercos",
			'04037' => "Chirivel",
			'04034' => "Cóbdar",
			'04035' => "Cuevas del Almanzora",
			'04038' => "Dalías",
			'04902' => "Ejido, El",
			'04041' => "Enix",
			'04043' => "Felix",
			'04044' => "Fines",
			'04045' => "Fiñana",
			'04046' => "Fondón",
			'04047' => "Gádor",
			'04048' => "Gallardos, Los",
			'04049' => "Garrucha",
			'04050' => "Gérgal",
			'04051' => "Huécija",
			'04052' => "Huércal de Almería",
			'04053' => "Huércal-Overa",
			'04054' => "Íllar",
			'04055' => "Instinción",
			'04056' => "Laroya",
			'04057' => "Láujar de Andarax",
			'04058' => "Líjar",
			'04059' => "Lubrín",
			'04060' => "Lucainena de las Torres",
			'04061' => "Lúcar",
			'04062' => "Macael",
			'04063' => "María",
			'04064' => "Mojácar",
			'04903' => "Mojonera, La",
			'04065' => "Nacimiento",
			'04066' => "Níjar",
			'04067' => "Ohanes",
			'04068' => "Olula de Castro",
			'04069' => "Olula del Río",
			'04070' => "Oria",
			'04071' => "Padules",
			'04072' => "Partaloa",
			'04073' => "Paterna del Río",
			'04074' => "Pechina",
			'04075' => "Pulpí",
			'04076' => "Purchena",
			'04077' => "Rágol",
			'04078' => "Rioja",
			'04079' => "Roquetas de Mar",
			'04080' => "Santa Cruz de Marchena",
			'04081' => "Santa Fe de Mondújar",
			'04082' => "Senés",
			'04083' => "Serón",
			'04084' => "Sierro",
			'04085' => "Somontín",
			'04086' => "Sorbas",
			'04087' => "Suflí",
			'04088' => "Tabernas",
			'04089' => "Taberno",
			'04090' => "Tahal",
			'04091' => "Terque",
			'04092' => "Tíjola",
			'04901' => "Tres Villas, Las",
			'04093' => "Turre",
			'04094' => "Turrillas",
			'04095' => "Uleila del Campo",
			'04096' => "Urrácal",
			'04097' => "Velefique",
			'04098' => "Vélez-Blanco",
			'04099' => "Vélez-Rubio",
			'04100' => "Vera",
			'04101' => "Viator",
			'04102' => "Vícar",
			'04103' => "Zurgena",
			'05001' => "Adanero",
			'05002' => "Adrada, La",
			'05005' => "Albornos",
			'05007' => "Aldeanueva de Santa Cruz",
			'05008' => "Aldeaseca",
			'05010' => "Aldehuela, La",
			'05012' => "Amavida",
			'05013' => "Arenal, El",
			'05014' => "Arenas de San Pedro",
			'05015' => "Arevalillo",
			'05016' => "Arévalo",
			'05017' => "Aveinte",
			'05018' => "Avellaneda",
			'05019' => "Ávila",
			'05021' => "Barco de Ávila, El",
			'05022' => "Barraco, El",
			'05023' => "Barromán",
			'05024' => "Becedas",
			'05025' => "Becedillas",
			'05026' => "Bercial de Zapardiel",
			'05027' => "Berlanas, Las",
			'05029' => "Bernuy-Zapardiel",
			'05030' => "Berrocalejo de Aragona",
			'05033' => "Blascomillán",
			'05034' => "Blasconuño de Matacabras",
			'05035' => "Blascosancho",
			'05036' => "Bohodón, El",
			'05037' => "Bohoyo",
			'05038' => "Bonilla de la Sierra",
			'05039' => "Brabos",
			'05040' => "Bularros",
			'05041' => "Burgohondo",
			'05042' => "Cabezas de Alambre",
			'05043' => "Cabezas del Pozo",
			'05044' => "Cabezas del Villar",
			'05045' => "Cabizuela",
			'05046' => "Canales",
			'05047' => "Candeleda",
			'05048' => "Cantiveros",
			'05049' => "Cardeñosa",
			'05051' => "Carrera, La",
			'05052' => "Casas del Puerto",
			'05053' => "Casasola",
			'05054' => "Casavieja",
			'05055' => "Casillas",
			'05056' => "Castellanos de Zapardiel",
			'05057' => "Cebreros",
			'05058' => "Cepeda la Mora",
			'05067' => "Chamartín",
			'05059' => "Cillán",
			'05060' => "Cisla",
			'05061' => "Colilla, La",
			'05062' => "Collado de Contreras",
			'05063' => "Collado del Mirón",
			'05064' => "Constanzana",
			'05065' => "Crespos",
			'05066' => "Cuevas del Valle",
			'05903' => "Diego del Carpio",
			'05069' => "Donjimeno",
			'05070' => "Donvidas",
			'05072' => "Espinosa de los Caballeros",
			'05073' => "Flores de Ávila",
			'05074' => "Fontiveros",
			'05075' => "Fresnedilla",
			'05076' => "Fresno, El",
			'05077' => "Fuente el Saúz",
			'05078' => "Fuentes de Año",
			'05079' => "Gallegos de Altamiros",
			'05080' => "Gallegos de Sobrinos",
			'05081' => "Garganta del Villar",
			'05082' => "Gavilanes",
			'05083' => "Gemuño",
			'05085' => "Gil García",
			'05084' => "Gilbuena",
			'05086' => "Gimialcón",
			'05087' => "Gotarrendura",
			'05088' => "Grandes y San Martín",
			'05089' => "Guisando",
			'05090' => "Gutierre-Muñoz",
			'05092' => "Hernansancho",
			'05093' => "Herradón de Pinares",
			'05094' => "Herreros de Suso",
			'05095' => "Higuera de las Dueñas",
			'05096' => "Hija de Dios, La",
			'05097' => "Horcajada, La",
			'05099' => "Horcajo de las Torres",
			'05100' => "Hornillo, El",
			'05102' => "Hoyo de Pinares, El",
			'05101' => "Hoyocasero",
			'05103' => "Hoyorredondo",
			'05106' => "Hoyos de Miguel Muñoz",
			'05104' => "Hoyos del Collado",
			'05105' => "Hoyos del Espino",
			'05107' => "Hurtumpascual",
			'05108' => "Junciana",
			'05109' => "Langa",
			'05110' => "Lanzahíta",
			'05113' => "Llanos de Tormes, Los",
			'05112' => "Losar del Barco, El",
			'05114' => "Madrigal de las Altas Torres",
			'05115' => "Maello",
			'05116' => "Malpartida de Corneja",
			'05117' => "Mamblas",
			'05118' => "Mancera de Arriba",
			'05119' => "Manjabálago y Ortigosa de Rioalmar",
			'05120' => "Marlín",
			'05121' => "Martiherrero",
			'05122' => "Martínez",
			'05123' => "Mediana de Voltoya",
			'05124' => "Medinilla",
			'05125' => "Mengamuñoz",
			'05126' => "Mesegar de Corneja",
			'05127' => "Mijares",
			'05128' => "Mingorría",
			'05129' => "Mirón, El",
			'05130' => "Mironcillo",
			'05131' => "Mirueña de los Infanzones",
			'05132' => "Mombeltrán",
			'05133' => "Monsalupe",
			'05134' => "Moraleja de Matacabras",
			'05135' => "Muñana",
			'05136' => "Muñico",
			'05138' => "Muñogalindo",
			'05139' => "Muñogrande",
			'05140' => "Muñomer del Peco",
			'05141' => "Muñopepe",
			'05142' => "Muñosancho",
			'05143' => "Muñotello",
			'05144' => "Narrillos del Álamo",
			'05145' => "Narrillos del Rebollar",
			'05149' => "Narros de Saldueña",
			'05147' => "Narros del Castillo",
			'05148' => "Narros del Puerto",
			'05152' => "Nava de Arévalo",
			'05153' => "Nava del Barco",
			'05151' => "Navacepedilla de Corneja",
			'05154' => "Navadijos",
			'05155' => "Navaescurial",
			'05156' => "Navahondilla",
			'05157' => "Navalacruz",
			'05158' => "Navalmoral",
			'05159' => "Navalonguilla",
			'05160' => "Navalosa",
			'05161' => "Navalperal de Pinares",
			'05162' => "Navalperal de Tormes",
			'05163' => "Navaluenga",
			'05164' => "Navaquesera",
			'05165' => "Navarredonda de Gredos",
			'05166' => "Navarredondilla",
			'05167' => "Navarrevisca",
			'05168' => "Navas del Marqués, Las",
			'05169' => "Navatalgordo",
			'05170' => "Navatejares",
			'05171' => "Neila de San Miguel",
			'05172' => "Niharra",
			'05173' => "Ojos-Albos",
			'05174' => "Orbita",
			'05175' => "Oso, El",
			'05176' => "Padiernos",
			'05177' => "Pajares de Adaja",
			'05178' => "Palacios de Goda",
			'05179' => "Papatrigo",
			'05180' => "Parral, El",
			'05181' => "Pascualcobo",
			'05182' => "Pedro Bernardo",
			'05183' => "Pedro-Rodríguez",
			'05184' => "Peguerinos",
			'05185' => "Peñalba de Ávila",
			'05186' => "Piedrahíta",
			'05187' => "Piedralaves",
			'05188' => "Poveda",
			'05189' => "Poyales del Hoyo",
			'05190' => "Pozanco",
			'05191' => "Pradosegar",
			'05192' => "Puerto Castilla",
			'05193' => "Rasueros",
			'05194' => "Riocabado",
			'05195' => "Riofrío",
			'05196' => "Rivilla de Barajas",
			'05197' => "Salobral",
			'05198' => "Salvadiós",
			'05199' => "San Bartolomé de Béjar",
			'05200' => "San Bartolomé de Corneja",
			'05201' => "San Bartolomé de Pinares",
			'05206' => "San Esteban de los Patos",
			'05208' => "San Esteban de Zapardiel",
			'05207' => "San Esteban del Valle",
			'05209' => "San García de Ingelmos",
			'05901' => "San Juan de Gredos",
			'05210' => "San Juan de la Encinilla",
			'05211' => "San Juan de la Nava",
			'05212' => "San Juan del Molinillo",
			'05213' => "San Juan del Olmo",
			'05214' => "San Lorenzo de Tormes",
			'05215' => "San Martín de la Vega del Alberche",
			'05216' => "San Martín del Pimpollar",
			'05217' => "San Miguel de Corneja",
			'05218' => "San Miguel de Serrezuela",
			'05219' => "San Pascual",
			'05220' => "San Pedro del Arroyo",
			'05231' => "San Vicente de Arévalo",
			'05204' => "Sanchidrián",
			'05205' => "Sanchorreja",
			'05222' => "Santa Cruz de Pinares",
			'05221' => "Santa Cruz del Valle",
			'05226' => "Santa María de los Caballeros",
			'05224' => "Santa María del Arroyo",
			'05225' => "Santa María del Berrocal",
			'05902' => "Santa María del Cubillo",
			'05227' => "Santa María del Tiétar",
			'05228' => "Santiago del Collado",
			'05904' => "Santiago del Tormes",
			'05229' => "Santo Domingo de las Posadas",
			'05230' => "Santo Tomé de Zabarcos",
			'05232' => "Serrada, La",
			'05233' => "Serranillos",
			'05234' => "Sigeres",
			'05235' => "Sinlabajos",
			'05236' => "Solana de Ávila",
			'05237' => "Solana de Rioalmar",
			'05238' => "Solosancho",
			'05239' => "Sotalbo",
			'05240' => "Sotillo de la Adrada",
			'05241' => "Tiemblo, El",
			'05242' => "Tiñosillos",
			'05243' => "Tolbaños",
			'05244' => "Tormellas",
			'05245' => "Tornadizos de Ávila",
			'05247' => "Torre, La",
			'05246' => "Tórtoles",
			'05249' => "Umbrías",
			'05251' => "Vadillo de la Sierra",
			'05252' => "Valdecasa",
			'05253' => "Vega de Santa María",
			'05254' => "Velayos",
			'05256' => "Villaflor",
			'05257' => "Villafranca de la Sierra",
			'05905' => "Villanueva de Ávila",
			'05258' => "Villanueva de Gómez",
			'05259' => "Villanueva del Aceral",
			'05260' => "Villanueva del Campillo",
			'05261' => "Villar de Corneja",
			'05262' => "Villarejo del Valle",
			'05263' => "Villatoro",
			'05264' => "Viñegra de Moraña",
			'05265' => "Vita",
			'05266' => "Zapardiel de la Cañada",
			'05267' => "Zapardiel de la Ribera",
			'06001' => "Acedera",
			'06002' => "Aceuchal",
			'06003' => "Ahillones",
			'06004' => "Alange",
			'06005' => "Albuera, La",
			'06006' => "Alburquerque",
			'06007' => "Alconchel",
			'06008' => "Alconera",
			'06009' => "Aljucén",
			'06010' => "Almendral",
			'06011' => "Almendralejo",
			'06012' => "Arroyo de San Serván",
			'06013' => "Atalaya",
			'06014' => "Azuaga",
			'06015' => "Badajoz",
			'06016' => "Barcarrota",
			'06017' => "Baterno",
			'06018' => "Benquerencia de la Serena",
			'06019' => "Berlanga",
			'06020' => "Bienvenida",
			'06021' => "Bodonal de la Sierra",
			'06022' => "Burguillos del Cerro",
			'06023' => "Cabeza del Buey",
			'06024' => "Cabeza la Vaca",
			'06025' => "Calamonte",
			'06026' => "Calera de León",
			'06027' => "Calzadilla de los Barros",
			'06028' => "Campanario",
			'06029' => "Campillo de Llerena",
			'06030' => "Capilla",
			'06031' => "Carmonita",
			'06032' => "Carrascalejo, El",
			'06033' => "Casas de Don Pedro",
			'06034' => "Casas de Reina",
			'06035' => "Castilblanco",
			'06036' => "Castuera",
			'06042' => "Cheles",
			'06037' => "Codosera, La",
			'06038' => "Cordobilla de Lácara",
			'06039' => "Coronada, La",
			'06040' => "Corte de Peleas",
			'06041' => "Cristina",
			'06043' => "Don Álvaro",
			'06044' => "Don Benito",
			'06045' => "Entrín Bajo",
			'06046' => "Esparragalejo",
			'06047' => "Esparragosa de la Serena",
			'06048' => "Esparragosa de Lares",
			'06049' => "Feria",
			'06050' => "Fregenal de la Sierra",
			'06051' => "Fuenlabrada de los Montes",
			'06052' => "Fuente de Cantos",
			'06053' => "Fuente del Arco",
			'06054' => "Fuente del Maestre",
			'06055' => "Fuentes de León",
			'06056' => "Garbayuela",
			'06057' => "Garlitos",
			'06058' => "Garrovilla, La",
			'06059' => "Granja de Torrehermosa",
			'06903' => "Guadiana",
			'06060' => "Guareña",
			'06061' => "Haba, La",
			'06062' => "Helechosa de los Montes",
			'06063' => "Herrera del Duque",
			'06064' => "Higuera de la Serena",
			'06065' => "Higuera de Llerena",
			'06066' => "Higuera de Vargas",
			'06067' => "Higuera la Real",
			'06068' => "Hinojosa del Valle",
			'06069' => "Hornachos",
			'06070' => "Jerez de los Caballeros",
			'06071' => "Lapa, La",
			'06073' => "Llera",
			'06074' => "Llerena",
			'06072' => "Lobón",
			'06075' => "Magacela",
			'06076' => "Maguilla",
			'06077' => "Malcocinado",
			'06078' => "Malpartida de la Serena",
			'06079' => "Manchita",
			'06080' => "Medellín",
			'06081' => "Medina de las Torres",
			'06082' => "Mengabril",
			'06083' => "Mérida",
			'06084' => "Mirandilla",
			'06085' => "Monesterio",
			'06086' => "Montemolín",
			'06087' => "Monterrubio de la Serena",
			'06088' => "Montijo",
			'06089' => "Morera, La",
			'06090' => "Nava de Santiago, La",
			'06091' => "Navalvillar de Pela",
			'06092' => "Nogales",
			'06093' => "Oliva de la Frontera",
			'06094' => "Oliva de Mérida",
			'06095' => "Olivenza",
			'06096' => "Orellana de la Sierra",
			'06097' => "Orellana la Vieja",
			'06098' => "Palomas",
			'06099' => "Parra, La",
			'06100' => "Peñalsordo",
			'06101' => "Peraleda del Zaucejo",
			'06102' => "Puebla de Alcocer",
			'06103' => "Puebla de la Calzada",
			'06104' => "Puebla de la Reina",
			'06107' => "Puebla de Obando",
			'06108' => "Puebla de Sancho Pérez",
			'06105' => "Puebla del Maestre",
			'06106' => "Puebla del Prior",
			'06902' => "Pueblonuevo del Guadiana",
			'06109' => "Quintana de la Serena",
			'06110' => "Reina",
			'06111' => "Rena",
			'06112' => "Retamal de Llerena",
			'06113' => "Ribera del Fresno",
			'06114' => "Risco",
			'06115' => "Roca de la Sierra, La",
			'06116' => "Salvaleón",
			'06117' => "Salvatierra de los Barros",
			'06119' => "San Pedro de Mérida",
			'06123' => "San Vicente de Alcántara",
			'06118' => "Sancti-Spíritus",
			'06120' => "Santa Amalia",
			'06121' => "Santa Marta",
			'06122' => "Santos de Maimona, Los",
			'06124' => "Segura de León",
			'06125' => "Siruela",
			'06126' => "Solana de los Barros",
			'06127' => "Talarrubias",
			'06128' => "Talavera la Real",
			'06129' => "Táliga",
			'06130' => "Tamurejo",
			'06131' => "Torre de Miguel Sesmero",
			'06132' => "Torremayor",
			'06133' => "Torremejía",
			'06134' => "Trasierra",
			'06135' => "Trujillanos",
			'06136' => "Usagre",
			'06137' => "Valdecaballeros",
			'06901' => "Valdelacalzada",
			'06138' => "Valdetorres",
			'06139' => "Valencia de las Torres",
			'06140' => "Valencia del Mombuey",
			'06141' => "Valencia del Ventoso",
			'06146' => "Valle de la Serena",
			'06147' => "Valle de Matamoros",
			'06148' => "Valle de Santa Ana",
			'06142' => "Valverde de Burguillos",
			'06143' => "Valverde de Leganés",
			'06144' => "Valverde de Llerena",
			'06145' => "Valverde de Mérida",
			'06149' => "Villafranca de los Barros",
			'06150' => "Villagarcía de la Torre",
			'06151' => "Villagonzalo",
			'06152' => "Villalba de los Barros",
			'06153' => "Villanueva de la Serena",
			'06154' => "Villanueva del Fresno",
			'06156' => "Villar de Rena",
			'06155' => "Villar del Rey",
			'06157' => "Villarta de los Montes",
			'06158' => "Zafra",
			'06159' => "Zahínos",
			'06160' => "Zalamea de la Serena",
			'06162' => "Zarza, La",
			'06161' => "Zarza-Capilla",
			'07002' => "Alaior",
			'07001' => "Alaró",
			'07003' => "Alcúdia",
			'07004' => "Algaida",
			'07005' => "Andratx",
			'07901' => "Ariany",
			'07006' => "Artà",
			'07007' => "Banyalbufar",
			'07008' => "Binissalem",
			'07009' => "Búger",
			'07010' => "Bunyola",
			'07011' => "Calvià",
			'07012' => "Campanet",
			'07013' => "Campos",
			'07014' => "Capdepera",
			'07064' => "Castell, Es",
			'07015' => "Ciutadella de Menorca",
			'07016' => "Consell",
			'07017' => "Costitx",
			'07018' => "Deià",
			'07026' => "Eivissa",
			'07019' => "Escorca",
			'07020' => "Esporles",
			'07021' => "Estellencs",
			'07022' => "Felanitx",
			'07023' => "Ferreries",
			'07024' => "Formentera",
			'07025' => "Fornalutx",
			'07027' => "Inca",
			'07028' => "Lloret de Vistalegre",
			'07029' => "Lloseta",
			'07030' => "Llubí",
			'07031' => "Llucmajor",
			'07033' => "Manacor",
			'07034' => "Mancor de la Vall",
			'07032' => "Maó",
			'07035' => "Maria de la Salut",
			'07036' => "Marratxí",
			'07037' => "Mercadal, Es",
			'07902' => "Migjorn Gran, Es",
			'07038' => "Montuïri",
			'07039' => "Muro",
			'07040' => "Palma",
			'07041' => "Petra",
			'07044' => "Pobla, Sa",
			'07042' => "Pollença",
			'07043' => "Porreres",
			'07045' => "Puigpunyent",
			'07059' => "Salines, Ses",
			'07046' => "Sant Antoni de Portmany",
			'07049' => "Sant Joan",
			'07050' => "Sant Joan de Labritja",
			'07048' => "Sant Josep de sa Talaia",
			'07051' => "Sant Llorenç des Cardassar",
			'07052' => "Sant Lluís",
			'07053' => "Santa Eugènia",
			'07054' => "Santa Eulària des Riu",
			'07055' => "Santa Margalida",
			'07056' => "Santa María del Camí",
			'07057' => "Santanyí",
			'07058' => "Selva",
			'07047' => "Sencelles",
			'07060' => "Sineu",
			'07061' => "Sóller",
			'07062' => "Son Servera",
			'07063' => "Valldemossa",
			'07065' => "Vilafranca de Bonany",
			'08001' => "Abrera",
			'08002' => "Aguilar de Segarra",
			'08014' => "Aiguafreda",
			'08003' => "Alella",
			'08004' => "Alpens",
			'08005' => "Ametlla del Vallès, L'",
			'08006' => "Arenys de Mar",
			'08007' => "Arenys de Munt",
			'08008' => "Argençola",
			'08009' => "Argentona",
			'08010' => "Artés",
			'08011' => "Avià",
			'08012' => "Avinyó",
			'08013' => "Avinyonet del Penedès",
			'08015' => "Badalona",
			'08904' => "Badia del Vallès",
			'08016' => "Bagà",
			'08017' => "Balenyà",
			'08018' => "Balsareny",
			'08252' => "Barberà del Vallès",
			'08019' => "Barcelona",
			'08020' => "Begues",
			'08021' => "Bellprat",
			'08022' => "Berga",
			'08023' => "Bigues i Riells del Fai",
			'08024' => "Borredà",
			'08025' => "Bruc, El",
			'08026' => "Brull, El",
			'08027' => "Cabanyes, Les",
			'08028' => "Cabrera d'Anoia",
			'08029' => "Cabrera de Mar",
			'08030' => "Cabrils",
			'08031' => "Calaf",
			'08034' => "Calders",
			'08033' => "Caldes de Montbui",
			'08032' => "Caldes d'Estrac",
			'08035' => "Calella",
			'08037' => "Calldetenes",
			'08038' => "Callús",
			'08036' => "Calonge de Segarra",
			'08039' => "Campins",
			'08040' => "Canet de Mar",
			'08041' => "Canovelles",
			'08042' => "Cànoves i Samalús",
			'08043' => "Canyelles",
			'08044' => "Capellades",
			'08045' => "Capolat",
			'08046' => "Cardedeu",
			'08047' => "Cardona",
			'08048' => "Carme",
			'08049' => "Casserres",
			'08057' => "Castell de l'Areny",
			'08052' => "Castellar de n'Hug",
			'08050' => "Castellar del Riu",
			'08051' => "Castellar del Vallès",
			'08053' => "Castellbell i el Vilar",
			'08054' => "Castellbisbal",
			'08055' => "Castellcir",
			'08056' => "Castelldefels",
			'08058' => "Castellet i la Gornal",
			'08060' => "Castellfollit de Riubregós",
			'08059' => "Castellfollit del Boix",
			'08061' => "Castellgalí",
			'08062' => "Castellnou de Bages",
			'08063' => "Castellolí",
			'08064' => "Castellterçol",
			'08065' => "Castellví de la Marca",
			'08066' => "Castellví de Rosanes",
			'08067' => "Centelles",
			'08268' => "Cercs",
			'08266' => "Cerdanyola del Vallès",
			'08068' => "Cervelló",
			'08069' => "Collbató",
			'08070' => "Collsuspina",
			'08071' => "Copons",
			'08072' => "Corbera de Llobregat",
			'08073' => "Cornellà de Llobregat",
			'08074' => "Cubelles",
			'08075' => "Dosrius",
			'08076' => "Esparreguera",
			'08077' => "Esplugues de Llobregat",
			'08078' => "Espunyola, L'",
			'08254' => "Esquirol, L'",
			'08079' => "Estany, L'",
			'08134' => "Figaró-Montmany",
			'08080' => "Fígols",
			'08082' => "Fogars de la Selva",
			'08081' => "Fogars de Montclús",
			'08083' => "Folgueroles",
			'08084' => "Fonollosa",
			'08085' => "Font-rubí",
			'08086' => "Franqueses del Vallès, Les",
			'08090' => "Gaià",
			'08087' => "Gallifa",
			'08088' => "Garriga, La",
			'08089' => "Gavà",
			'08091' => "Gelida",
			'08092' => "Gironella",
			'08093' => "Gisclareny",
			'08094' => "Granada, La",
			'08095' => "Granera",
			'08096' => "Granollers",
			'08097' => "Gualba",
			'08099' => "Guardiola de Berguedà",
			'08100' => "Gurb",
			'08101' => "Hospitalet de Llobregat, L'",
			'08162' => "Hostalets de Pierola, Els",
			'08102' => "Igualada",
			'08103' => "Jorba",
			'08104' => "Llacuna, La",
			'08105' => "Llagosta, La",
			'08107' => "Lliçà d'Amunt",
			'08108' => "Lliçà de Vall",
			'08106' => "Llinars del Vallès",
			'08109' => "Lluçà",
			'08110' => "Malgrat de Mar",
			'08111' => "Malla",
			'08112' => "Manlleu",
			'08113' => "Manresa",
			'08242' => "Marganell",
			'08114' => "Martorell",
			'08115' => "Martorelles",
			'08116' => "Masies de Roda, Les",
			'08117' => "Masies de Voltregà, Les",
			'08118' => "Masnou, El",
			'08119' => "Masquefa",
			'08120' => "Matadepera",
			'08121' => "Mataró",
			'08122' => "Mediona",
			'08138' => "Moià",
			'08123' => "Molins de Rei",
			'08124' => "Mollet del Vallès",
			'08128' => "Monistrol de Calders",
			'08127' => "Monistrol de Montserrat",
			'08125' => "Montcada i Reixac",
			'08130' => "Montclar",
			'08131' => "Montesquiu",
			'08126' => "Montgat",
			'08132' => "Montmajor",
			'08133' => "Montmaneu",
			'08135' => "Montmeló",
			'08136' => "Montornès del Vallès",
			'08137' => "Montseny",
			'08129' => "Muntanyola",
			'08139' => "Mura",
			'08140' => "Navarcles",
			'08141' => "Navàs",
			'08142' => "Nou de Berguedà, La",
			'08143' => "Òdena",
			'08145' => "Olèrdola",
			'08146' => "Olesa de Bonesvalls",
			'08147' => "Olesa de Montserrat",
			'08148' => "Olivella",
			'08149' => "Olost",
			'08144' => "Olvan",
			'08150' => "Orís",
			'08151' => "Oristà",
			'08152' => "Orpí",
			'08153' => "Òrrius",
			'08154' => "Pacs del Penedès",
			'08155' => "Palafolls",
			'08156' => "Palau-solità i Plegamans",
			'08157' => "Pallejà",
			'08905' => "Palma de Cervelló, La",
			'08158' => "Papiol, El",
			'08159' => "Parets del Vallès",
			'08160' => "Perafita",
			'08161' => "Piera",
			'08163' => "Pineda de Mar",
			'08164' => "Pla del Penedès, El",
			'08165' => "Pobla de Claramunt, La",
			'08166' => "Pobla de Lillet, La",
			'08167' => "Polinyà",
			'08182' => "Pont de Vilomara i Rocafort, El",
			'08168' => "Pontons",
			'08169' => "Prat de Llobregat, El",
			'08171' => "Prats de Lluçanès",
			'08170' => "Prats de Rei, Els",
			'08230' => "Premià de Dalt",
			'08172' => "Premià de Mar",
			'08174' => "Puigdàlber",
			'08175' => "Puig-reig",
			'08176' => "Pujalt",
			'08177' => "Quar, La",
			'08178' => "Rajadell",
			'08179' => "Rellinars",
			'08180' => "Ripollet",
			'08181' => "Roca del Vallès, La",
			'08183' => "Roda de Ter",
			'08184' => "Rubí",
			'08185' => "Rubió",
			'08901' => "Rupit i Pruit",
			'08187' => "Sabadell",
			'08188' => "Sagàs",
			'08190' => "Saldes",
			'08191' => "Sallent",
			'08194' => "Sant Adrià de Besòs",
			'08195' => "Sant Agustí de Lluçanès",
			'08196' => "Sant Andreu de la Barca",
			'08197' => "Sant Andreu de Llavaneres",
			'08198' => "Sant Antoni de Vilamajor",
			'08199' => "Sant Bartomeu del Grau",
			'08200' => "Sant Boi de Llobregat",
			'08201' => "Sant Boi de Lluçanès",
			'08203' => "Sant Cebrià de Vallalta",
			'08202' => "Sant Celoni",
			'08204' => "Sant Climent de Llobregat",
			'08205' => "Sant Cugat del Vallès",
			'08206' => "Sant Cugat Sesgarrigues",
			'08207' => "Sant Esteve de Palautordera",
			'08208' => "Sant Esteve Sesrovires",
			'08210' => "Sant Feliu de Codines",
			'08211' => "Sant Feliu de Llobregat",
			'08212' => "Sant Feliu Sasserra",
			'08209' => "Sant Fost de Campsentelles",
			'08213' => "Sant Fruitós de Bages",
			'08215' => "Sant Hipòlit de Voltregà",
			'08193' => "Sant Iscle de Vallalta",
			'08216' => "Sant Jaume de Frontanyà",
			'08218' => "Sant Joan de Vilatorrada",
			'08217' => "Sant Joan Despí",
			'08903' => "Sant Julià de Cerdanyola",
			'08220' => "Sant Julià de Vilatorta",
			'08221' => "Sant Just Desvern",
			'08222' => "Sant Llorenç d'Hortons",
			'08223' => "Sant Llorenç Savall",
			'08225' => "Sant Martí d'Albars",
			'08224' => "Sant Martí de Centelles",
			'08226' => "Sant Martí de Tous",
			'08227' => "Sant Martí Sarroca",
			'08228' => "Sant Martí Sesgueioles",
			'08229' => "Sant Mateu de Bages",
			'08231' => "Sant Pere de Ribes",
			'08232' => "Sant Pere de Riudebitlles",
			'08233' => "Sant Pere de Torelló",
			'08234' => "Sant Pere de Vilamajor",
			'08189' => "Sant Pere Sallavinera",
			'08235' => "Sant Pol de Mar",
			'08236' => "Sant Quintí de Mediona",
			'08237' => "Sant Quirze de Besora",
			'08238' => "Sant Quirze del Vallès",
			'08239' => "Sant Quirze Safaja",
			'08240' => "Sant Sadurní d'Anoia",
			'08241' => "Sant Sadurní d'Osormort",
			'08098' => "Sant Salvador de Guardiola",
			'08262' => "Sant Vicenç de Castellet",
			'08264' => "Sant Vicenç de Montalt",
			'08265' => "Sant Vicenç de Torelló",
			'08263' => "Sant Vicenç dels Horts",
			'08243' => "Santa Cecília de Voltregà",
			'08244' => "Santa Coloma de Cervelló",
			'08245' => "Santa Coloma de Gramenet",
			'08246' => "Santa Eugènia de Berga",
			'08247' => "Santa Eulàlia de Riuprimer",
			'08248' => "Santa Eulàlia de Ronçana",
			'08249' => "Santa Fe del Penedès",
			'08250' => "Santa Margarida de Montbui",
			'08251' => "Santa Margarida i els Monjos",
			'08253' => "Santa Maria de Besora",
			'08256' => "Santa Maria de Martorelles",
			'08255' => "Santa Maria de Merlès",
			'08257' => "Santa Maria de Miralles",
			'08259' => "Santa Maria de Palautordera",
			'08258' => "Santa Maria d'Oló",
			'08260' => "Santa Perpètua de Mogoda",
			'08261' => "Santa Susanna",
			'08192' => "Santpedor",
			'08267' => "Sentmenat",
			'08269' => "Seva",
			'08270' => "Sitges",
			'08271' => "Sobremunt",
			'08272' => "Sora",
			'08273' => "Subirats",
			'08274' => "Súria",
			'08276' => "Tagamanent",
			'08277' => "Talamanca",
			'08278' => "Taradell",
			'08275' => "Tavèrnoles",
			'08280' => "Tavertet",
			'08281' => "Teià",
			'08279' => "Terrassa",
			'08282' => "Tiana",
			'08283' => "Tona",
			'08284' => "Tordera",
			'08285' => "Torelló",
			'08286' => "Torre de Claramunt, La",
			'08287' => "Torrelavit",
			'08288' => "Torrelles de Foix",
			'08289' => "Torrelles de Llobregat",
			'08290' => "Ullastrell",
			'08291' => "Vacarisses",
			'08292' => "Vallbona d'Anoia",
			'08293' => "Vallcebre",
			'08294' => "Vallgorguina",
			'08295' => "Vallirana",
			'08296' => "Vallromanes",
			'08297' => "Veciana",
			'08298' => "Vic",
			'08299' => "Vilada",
			'08301' => "Viladecans",
			'08300' => "Viladecavalls",
			'08305' => "Vilafranca del Penedès",
			'08306' => "Vilalba Sasserra",
			'08303' => "Vilanova de Sau",
			'08302' => "Vilanova del Camí",
			'08902' => "Vilanova del Vallès",
			'08307' => "Vilanova i la Geltrú",
			'08214' => "Vilassar de Dalt",
			'08219' => "Vilassar de Mar",
			'08304' => "Vilobí del Penedès",
			'08308' => "Viver i Serrateix",
			'09001' => "Abajas",
			'09003' => "Adrada de Haza",
			'09006' => "Aguas Cándidas",
			'09007' => "Aguilar de Bureba",
			'09009' => "Albillos",
			'09010' => "Alcocero de Mola",
			'09011' => "Alfoz de Bricia",
			'09907' => "Alfoz de Quintanadueñas",
			'09012' => "Alfoz de Santa Gadea",
			'09013' => "Altable",
			'09014' => "Altos, Los",
			'09016' => "Ameyugo",
			'09017' => "Anguix",
			'09018' => "Aranda de Duero",
			'09019' => "Arandilla",
			'09020' => "Arauzo de Miel",
			'09021' => "Arauzo de Salce",
			'09022' => "Arauzo de Torre",
			'09023' => "Arcos",
			'09024' => "Arenillas de Riopisuerga",
			'09025' => "Arija",
			'09026' => "Arlanzón",
			'09027' => "Arraya de Oca",
			'09029' => "Atapuerca",
			'09030' => "Ausines, Los",
			'09032' => "Avellanosa de Muñó",
			'09033' => "Bahabón de Esgueva",
			'09034' => "Balbases, Los",
			'09035' => "Baños de Valdearados",
			'09036' => "Bañuelos de Bureba",
			'09037' => "Barbadillo de Herreros",
			'09038' => "Barbadillo del Mercado",
			'09039' => "Barbadillo del Pez",
			'09041' => "Barrio de Muñó",
			'09043' => "Barrios de Bureba, Los",
			'09044' => "Barrios de Colina",
			'09045' => "Basconcillos del Tozo",
			'09046' => "Bascuñana",
			'09047' => "Belbimbre",
			'09048' => "Belorado",
			'09050' => "Berberana",
			'09051' => "Berlangas de Roa",
			'09052' => "Berzosa de Bureba",
			'09054' => "Bozoó",
			'09055' => "Brazacorta",
			'09056' => "Briviesca",
			'09057' => "Bugedo",
			'09058' => "Buniel",
			'09059' => "Burgos",
			'09060' => "Busto de Bureba",
			'09061' => "Cabañes de Esgueva",
			'09062' => "Cabezón de la Sierra",
			'09064' => "Caleruega",
			'09065' => "Campillo de Aranda",
			'09066' => "Campolara",
			'09067' => "Canicosa de la Sierra",
			'09068' => "Cantabrana",
			'09070' => "Carazo",
			'09071' => "Carcedo de Bureba",
			'09072' => "Carcedo de Burgos",
			'09073' => "Cardeñadijo",
			'09074' => "Cardeñajimeno",
			'09075' => "Cardeñuela Riopico",
			'09076' => "Carrias",
			'09077' => "Cascajares de Bureba",
			'09078' => "Cascajares de la Sierra",
			'09079' => "Castellanos de Castro",
			'09083' => "Castil de Peones",
			'09082' => "Castildelgado",
			'09084' => "Castrillo de la Reina",
			'09085' => "Castrillo de la Vega",
			'09088' => "Castrillo de Riopisuerga",
			'09086' => "Castrillo del Val",
			'09090' => "Castrillo Mota de Judíos",
			'09091' => "Castrojeriz",
			'09063' => "Cavia",
			'09093' => "Cayuela",
			'09094' => "Cebrecos",
			'09095' => "Celada del Camino",
			'09098' => "Cerezo de Río Tirón",
			'09100' => "Cerratón de Juarros",
			'09101' => "Ciadoncha",
			'09102' => "Cillaperlata",
			'09103' => "Cilleruelo de Abajo",
			'09104' => "Cilleruelo de Arriba",
			'09105' => "Ciruelos de Cervera",
			'09108' => "Cogollos",
			'09109' => "Condado de Treviño",
			'09110' => "Contreras",
			'09112' => "Coruña del Conde",
			'09113' => "Covarrubias",
			'09114' => "Cubillo del Campo",
			'09115' => "Cubo de Bureba",
			'09117' => "Cueva de Roa, La",
			'09119' => "Cuevas de San Clemente",
			'09120' => "Encío",
			'09122' => "Espinosa de Cervera",
			'09124' => "Espinosa de los Monteros",
			'09123' => "Espinosa del Camino",
			'09125' => "Estépar",
			'09127' => "Fontioso",
			'09128' => "Frandovínez",
			'09129' => "Fresneda de la Sierra Tirón",
			'09130' => "Fresneña",
			'09131' => "Fresnillo de las Dueñas",
			'09132' => "Fresno de Río Tirón",
			'09133' => "Fresno de Rodilla",
			'09134' => "Frías",
			'09135' => "Fuentebureba",
			'09136' => "Fuentecén",
			'09137' => "Fuentelcésped",
			'09138' => "Fuentelisendo",
			'09139' => "Fuentemolinos",
			'09140' => "Fuentenebro",
			'09141' => "Fuentespina",
			'09143' => "Galbarros",
			'09144' => "Gallega, La",
			'09148' => "Grijalba",
			'09149' => "Grisaleña",
			'09151' => "Gumiel de Izán",
			'09152' => "Gumiel de Mercado",
			'09154' => "Hacinas",
			'09155' => "Haza",
			'09159' => "Hontanas",
			'09160' => "Hontangas",
			'09162' => "Hontoria de la Cantera",
			'09164' => "Hontoria de Valdearados",
			'09163' => "Hontoria del Pinar",
			'09166' => "Hormazas, Las",
			'09167' => "Hornillos del Camino",
			'09168' => "Horra, La",
			'09169' => "Hortigüela",
			'09170' => "Hoyales de Roa",
			'09172' => "Huérmeces",
			'09173' => "Huerta de Arriba",
			'09174' => "Huerta de Rey",
			'09175' => "Humada",
			'09176' => "Hurones",
			'09177' => "Ibeas de Juarros",
			'09178' => "Ibrillos",
			'09179' => "Iglesiarrubia",
			'09180' => "Iglesias",
			'09181' => "Isar",
			'09182' => "Itero del Castillo",
			'09183' => "Jaramillo de la Fuente",
			'09184' => "Jaramillo Quemado",
			'09189' => "Junta de Traslaloma",
			'09190' => "Junta de Villalba de Losa",
			'09191' => "Jurisdicción de Lara",
			'09192' => "Jurisdicción de San Zadornil",
			'09194' => "Lerma",
			'09195' => "Llano de Bureba",
			'09196' => "Madrigal del Monte",
			'09197' => "Madrigalejo del Monte",
			'09198' => "Mahamud",
			'09199' => "Mambrilla de Castrejón",
			'09200' => "Mambrillas de Lara",
			'09201' => "Mamolar",
			'09202' => "Manciles",
			'09206' => "Mazuela",
			'09208' => "Mecerreyes",
			'09209' => "Medina de Pomar",
			'09211' => "Melgar de Fernamental",
			'09213' => "Merindad de Cuesta-Urria",
			'09214' => "Merindad de Montija",
			'09906' => "Merindad de Río Ubierna",
			'09215' => "Merindad de Sotoscueva",
			'09216' => "Merindad de Valdeporres",
			'09217' => "Merindad de Valdivielso",
			'09218' => "Milagros",
			'09219' => "Miranda de Ebro",
			'09220' => "Miraveche",
			'09221' => "Modúbar de la Emparedada",
			'09223' => "Monasterio de la Sierra",
			'09224' => "Monasterio de Rodilla",
			'09225' => "Moncalvillo",
			'09226' => "Monterrubio de la Demanda",
			'09227' => "Montorio",
			'09228' => "Moradillo de Roa",
			'09229' => "Nava de Roa",
			'09230' => "Navas de Bureba",
			'09231' => "Nebreda",
			'09232' => "Neila",
			'09235' => "Olmedillo de Roa",
			'09236' => "Olmillos de Muñó",
			'09238' => "Oña",
			'09239' => "Oquillas",
			'09241' => "Orbaneja Riopico",
			'09242' => "Padilla de Abajo",
			'09243' => "Padilla de Arriba",
			'09244' => "Padrones de Bureba",
			'09246' => "Palacios de la Sierra",
			'09247' => "Palacios de Riopisuerga",
			'09248' => "Palazuelos de la Sierra",
			'09249' => "Palazuelos de Muñó",
			'09250' => "Pampliega",
			'09251' => "Pancorbo",
			'09253' => "Pardilla",
			'09255' => "Partido de la Sierra en Tobalina",
			'09256' => "Pedrosa de Duero",
			'09259' => "Pedrosa de Río Úrbel",
			'09257' => "Pedrosa del Páramo",
			'09258' => "Pedrosa del Príncipe",
			'09261' => "Peñaranda de Duero",
			'09262' => "Peral de Arlanza",
			'09265' => "Piérnigas",
			'09266' => "Pineda de la Sierra",
			'09267' => "Pineda Trasmonte",
			'09268' => "Pinilla de los Barruecos",
			'09269' => "Pinilla de los Moros",
			'09270' => "Pinilla Trasmonte",
			'09272' => "Poza de la Sal",
			'09273' => "Prádanos de Bureba",
			'09274' => "Pradoluengo",
			'09275' => "Presencio",
			'09276' => "Puebla de Arganzón, La",
			'09277' => "Puentedura",
			'09279' => "Quemada",
			'09281' => "Quintana del Pidio",
			'09280' => "Quintanabureba",
			'09283' => "Quintanaélez",
			'09287' => "Quintanaortuño",
			'09288' => "Quintanapalla",
			'09289' => "Quintanar de la Sierra",
			'09292' => "Quintanavides",
			'09294' => "Quintanilla de la Mata",
			'09901' => "Quintanilla del Agua y Tordueles",
			'09295' => "Quintanilla del Coco",
			'09298' => "Quintanilla San García",
			'09301' => "Quintanilla Vivar",
			'09297' => "Quintanillas, Las",
			'09302' => "Rabanera del Pinar",
			'09303' => "Rábanos",
			'09304' => "Rabé de las Calzadas",
			'09306' => "Rebolledo de la Torre",
			'09307' => "Redecilla del Camino",
			'09308' => "Redecilla del Campo",
			'09309' => "Regumiel de la Sierra",
			'09310' => "Reinoso",
			'09311' => "Retuerta",
			'09314' => "Revilla del Campo",
			'09316' => "Revilla Vallejera",
			'09312' => "Revilla y Ahedo, La",
			'09315' => "Revillarruz",
			'09317' => "Rezmondo",
			'09318' => "Riocavado de la Sierra",
			'09321' => "Roa",
			'09323' => "Rojas",
			'09325' => "Royuela de Río Franco",
			'09326' => "Rubena",
			'09327' => "Rublacedo de Abajo",
			'09328' => "Rucandio",
			'09329' => "Salas de Bureba",
			'09330' => "Salas de los Infantes",
			'09332' => "Saldaña de Burgos",
			'09334' => "Salinillas de Bureba",
			'09335' => "San Adrián de Juarros",
			'09337' => "San Juan del Monte",
			'09338' => "San Mamés de Burgos",
			'09339' => "San Martín de Rubiales",
			'09340' => "San Millán de Lara",
			'09360' => "San Vicente del Valle",
			'09343' => "Santa Cecilia",
			'09345' => "Santa Cruz de la Salceda",
			'09346' => "Santa Cruz del Valle Urbión",
			'09347' => "Santa Gadea del Cid",
			'09348' => "Santa Inés",
			'09350' => "Santa María del Campo",
			'09351' => "Santa María del Invierno",
			'09352' => "Santa María del Mercadillo",
			'09353' => "Santa María Ribarredonda",
			'09354' => "Santa Olalla de Bureba",
			'09355' => "Santibáñez de Esgueva",
			'09356' => "Santibáñez del Val",
			'09358' => "Santo Domingo de Silos",
			'09361' => "Sargentes de la Lora",
			'09362' => "Sarracín",
			'09363' => "Sasamón",
			'09365' => "Sequera de Haza, La",
			'09366' => "Solarana",
			'09368' => "Sordillos",
			'09369' => "Sotillo de la Ribera",
			'09372' => "Sotragero",
			'09373' => "Sotresgudo",
			'09374' => "Susinos del Páramo",
			'09375' => "Tamarón",
			'09377' => "Tardajos",
			'09378' => "Tejada",
			'09380' => "Terradillos de Esgueva",
			'09381' => "Tinieblas de la Sierra",
			'09382' => "Tobar",
			'09384' => "Tordómar",
			'09386' => "Torrecilla del Monte",
			'09387' => "Torregalindo",
			'09388' => "Torrelara",
			'09389' => "Torrepadre",
			'09390' => "Torresandino",
			'09391' => "Tórtoles de Esgueva",
			'09392' => "Tosantos",
			'09394' => "Trespaderne",
			'09395' => "Tubilla del Agua",
			'09396' => "Tubilla del Lago",
			'09398' => "Úrbel del Castillo",
			'09400' => "Vadocondes",
			'09403' => "Valdeande",
			'09405' => "Valdezate",
			'09406' => "Valdorros",
			'09408' => "Vallarta de Bureba",
			'09904' => "Valle de las Navas",
			'09908' => "Valle de Losa",
			'09409' => "Valle de Manzanedo",
			'09410' => "Valle de Mena",
			'09411' => "Valle de Oca",
			'09902' => "Valle de Santibáñez",
			'09905' => "Valle de Sedano",
			'09412' => "Valle de Tobalina",
			'09413' => "Valle de Valdebezana",
			'09414' => "Valle de Valdelaguna",
			'09415' => "Valle de Valdelucio",
			'09416' => "Valle de Zamanzas",
			'09417' => "Vallejera",
			'09418' => "Valles de Palenzuela",
			'09419' => "Valluércanes",
			'09407' => "Valmala",
			'09422' => "Vid de Bureba, La",
			'09421' => "Vid y Barrios, La",
			'09423' => "Vileña",
			'09427' => "Villadiego",
			'09428' => "Villaescusa de Roa",
			'09429' => "Villaescusa la Sombría",
			'09430' => "Villaespasa",
			'09431' => "Villafranca Montes de Oca",
			'09432' => "Villafruela",
			'09433' => "Villagalijo",
			'09434' => "Villagonzalo Pedernales",
			'09437' => "Villahoz",
			'09438' => "Villalba de Duero",
			'09439' => "Villalbilla de Burgos",
			'09440' => "Villalbilla de Gumiel",
			'09441' => "Villaldemiro",
			'09442' => "Villalmanzo",
			'09443' => "Villamayor de los Montes",
			'09444' => "Villamayor de Treviño",
			'09445' => "Villambistia",
			'09446' => "Villamedianilla",
			'09447' => "Villamiel de la Sierra",
			'09448' => "Villangómez",
			'09449' => "Villanueva de Argaño",
			'09450' => "Villanueva de Carazo",
			'09451' => "Villanueva de Gumiel",
			'09454' => "Villanueva de Teba",
			'09455' => "Villaquirán de la Puebla",
			'09456' => "Villaquirán de los Infantes",
			'09903' => "Villarcayo de Merindad de Castilla la Vieja",
			'09458' => "Villariezo",
			'09460' => "Villasandino",
			'09463' => "Villasur de Herreros",
			'09464' => "Villatuelda",
			'09466' => "Villaverde del Monte",
			'09467' => "Villaverde-Mogina",
			'09471' => "Villayerno Morquillas",
			'09472' => "Villazopeque",
			'09473' => "Villegas",
			'09476' => "Villoruebo",
			'09424' => "Viloria de Rioja",
			'09425' => "Vilviestre del Pinar",
			'09478' => "Vizcaínos",
			'09480' => "Zael",
			'09482' => "Zarzosa de Río Pisuerga",
			'09483' => "Zazuar",
			'09485' => "Zuñeda",
			'10001' => "Abadía",
			'10002' => "Abertura",
			'10003' => "Acebo",
			'10004' => "Acehúche",
			'10005' => "Aceituna",
			'10006' => "Ahigal",
			'10903' => "Alagón del Río",
			'10007' => "Albalá",
			'10008' => "Alcántara",
			'10009' => "Alcollarín",
			'10010' => "Alcuéscar",
			'10012' => "Aldea del Cano",
			'10013' => "Aldea del Obispo, La",
			'10011' => "Aldeacentenera",
			'10014' => "Aldeanueva de la Vera",
			'10015' => "Aldeanueva del Camino",
			'10016' => "Aldehuela de Jerte",
			'10017' => "Alía",
			'10018' => "Aliseda",
			'10019' => "Almaraz",
			'10020' => "Almoharín",
			'10021' => "Arroyo de la Luz",
			'10023' => "Arroyomolinos",
			'10022' => "Arroyomolinos de la Vera",
			'10024' => "Baños de Montemayor",
			'10025' => "Barrado",
			'10026' => "Belvís de Monroy",
			'10027' => "Benquerencia",
			'10028' => "Berrocalejo",
			'10029' => "Berzocana",
			'10030' => "Bohonal de Ibor",
			'10031' => "Botija",
			'10032' => "Brozas",
			'10033' => "Cabañas del Castillo",
			'10034' => "Cabezabellosa",
			'10035' => "Cabezuela del Valle",
			'10036' => "Cabrero",
			'10037' => "Cáceres",
			'10038' => "Cachorrilla",
			'10039' => "Cadalso",
			'10040' => "Calzadilla",
			'10041' => "Caminomorisco",
			'10042' => "Campillo de Deleitosa",
			'10043' => "Campo Lugar",
			'10044' => "Cañamero",
			'10045' => "Cañaveral",
			'10046' => "Carbajo",
			'10047' => "Carcaboso",
			'10048' => "Carrascalejo",
			'10049' => "Casar de Cáceres",
			'10050' => "Casar de Palomero",
			'10051' => "Casares de las Hurdes",
			'10052' => "Casas de Don Antonio",
			'10053' => "Casas de Don Gómez",
			'10056' => "Casas de Millán",
			'10057' => "Casas de Miravete",
			'10054' => "Casas del Castañar",
			'10055' => "Casas del Monte",
			'10058' => "Casatejada",
			'10059' => "Casillas de Coria",
			'10060' => "Castañar de Ibor",
			'10061' => "Ceclavín",
			'10062' => "Cedillo",
			'10063' => "Cerezo",
			'10064' => "Cilleros",
			'10065' => "Collado de la Vera",
			'10066' => "Conquista de la Sierra",
			'10067' => "Coria",
			'10068' => "Cuacos de Yuste",
			'10069' => "Cumbre, La",
			'10070' => "Deleitosa",
			'10071' => "Descargamaría",
			'10072' => "Eljas",
			'10073' => "Escurial",
			'10075' => "Fresnedoso de Ibor",
			'10076' => "Galisteo",
			'10077' => "Garciaz",
			'10079' => "Garganta la Olla",
			'10078' => "Garganta, La",
			'10080' => "Gargantilla",
			'10081' => "Gargüera",
			'10082' => "Garrovillas de Alconétar",
			'10083' => "Garvín",
			'10084' => "Gata",
			'10085' => "Gordo, El",
			'10086' => "Granja, La",
			'10087' => "Guadalupe",
			'10088' => "Guijo de Coria",
			'10089' => "Guijo de Galisteo",
			'10090' => "Guijo de Granadilla",
			'10091' => "Guijo de Santa Bárbara",
			'10092' => "Herguijuela",
			'10093' => "Hernán-Pérez",
			'10094' => "Herrera de Alcántara",
			'10095' => "Herreruela",
			'10096' => "Hervás",
			'10097' => "Higuera de Albalat",
			'10098' => "Hinojal",
			'10099' => "Holguera",
			'10100' => "Hoyos",
			'10101' => "Huélaga",
			'10102' => "Ibahernando",
			'10103' => "Jaraicejo",
			'10104' => "Jaraíz de la Vera",
			'10105' => "Jarandilla de la Vera",
			'10106' => "Jarilla",
			'10107' => "Jerte",
			'10108' => "Ladrillar",
			'10109' => "Logrosán",
			'10110' => "Losar de la Vera",
			'10111' => "Madrigal de la Vera",
			'10112' => "Madrigalejo",
			'10113' => "Madroñera",
			'10114' => "Majadas",
			'10115' => "Malpartida de Cáceres",
			'10116' => "Malpartida de Plasencia",
			'10117' => "Marchagaz",
			'10118' => "Mata de Alcántara",
			'10119' => "Membrío",
			'10120' => "Mesas de Ibor",
			'10121' => "Miajadas",
			'10122' => "Millanes",
			'10123' => "Mirabel",
			'10124' => "Mohedas de Granadilla",
			'10125' => "Monroy",
			'10126' => "Montánchez",
			'10127' => "Montehermoso",
			'10128' => "Moraleja",
			'10129' => "Morcillo",
			'10130' => "Navaconcejo",
			'10131' => "Navalmoral de la Mata",
			'10132' => "Navalvillar de Ibor",
			'10133' => "Navas del Madroño",
			'10134' => "Navezuelas",
			'10135' => "Nuñomoral",
			'10136' => "Oliva de Plasencia",
			'10137' => "Palomero",
			'10138' => "Pasarón de la Vera",
			'10139' => "Pedroso de Acim",
			'10140' => "Peraleda de la Mata",
			'10141' => "Peraleda de San Román",
			'10142' => "Perales del Puerto",
			'10143' => "Pescueza",
			'10144' => "Pesga, La",
			'10145' => "Piedras Albas",
			'10146' => "Pinofranqueado",
			'10147' => "Piornal",
			'10148' => "Plasencia",
			'10149' => "Plasenzuela",
			'10150' => "Portaje",
			'10151' => "Portezuelo",
			'10152' => "Pozuelo de Zarzón",
			'10905' => "Pueblonuevo de Miramontes",
			'10153' => "Puerto de Santa Cruz",
			'10154' => "Rebollar",
			'10155' => "Riolobos",
			'10156' => "Robledillo de Gata",
			'10157' => "Robledillo de la Vera",
			'10158' => "Robledillo de Trujillo",
			'10159' => "Robledollano",
			'10160' => "Romangordo",
			'10901' => "Rosalejo",
			'10161' => "Ruanes",
			'10162' => "Salorino",
			'10163' => "Salvatierra de Santiago",
			'10164' => "San Martín de Trevejo",
			'10165' => "Santa Ana",
			'10166' => "Santa Cruz de la Sierra",
			'10167' => "Santa Cruz de Paniagua",
			'10168' => "Santa Marta de Magasca",
			'10169' => "Santiago de Alcántara",
			'10170' => "Santiago del Campo",
			'10171' => "Santibáñez el Alto",
			'10172' => "Santibáñez el Bajo",
			'10173' => "Saucedilla",
			'10174' => "Segura de Toro",
			'10175' => "Serradilla",
			'10176' => "Serrejón",
			'10177' => "Sierra de Fuentes",
			'10178' => "Talaván",
			'10179' => "Talaveruela de la Vera",
			'10180' => "Talayuela",
			'10181' => "Tejeda de Tiétar",
			'10904' => "Tiétar",
			'10182' => "Toril",
			'10183' => "Tornavacas",
			'10184' => "Torno, El",
			'10187' => "Torre de Don Miguel",
			'10188' => "Torre de Santa María",
			'10185' => "Torrecilla de los Ángeles",
			'10186' => "Torrecillas de la Tiesa",
			'10190' => "Torrejón el Rubio",
			'10189' => "Torrejoncillo",
			'10191' => "Torremenga",
			'10192' => "Torremocha",
			'10193' => "Torreorgaz",
			'10194' => "Torrequemada",
			'10195' => "Trujillo",
			'10196' => "Valdastillas",
			'10197' => "Valdecañas de Tajo",
			'10198' => "Valdefuentes",
			'10199' => "Valdehúncar",
			'10200' => "Valdelacasa de Tajo",
			'10201' => "Valdemorales",
			'10202' => "Valdeobispo",
			'10203' => "Valencia de Alcántara",
			'10204' => "Valverde de la Vera",
			'10205' => "Valverde del Fresno",
			'10902' => "Vegaviana",
			'10206' => "Viandar de la Vera",
			'10207' => "Villa del Campo",
			'10208' => "Villa del Rey",
			'10209' => "Villamesías",
			'10210' => "Villamiel",
			'10211' => "Villanueva de la Sierra",
			'10212' => "Villanueva de la Vera",
			'10214' => "Villar de Plasencia",
			'10213' => "Villar del Pedroso",
			'10215' => "Villasbuenas de Gata",
			'10216' => "Zarza de Granadilla",
			'10217' => "Zarza de Montánchez",
			'10218' => "Zarza la Mayor",
			'10219' => "Zorita",
			'11001' => "Alcalá de los Gazules",
			'11002' => "Alcalá del Valle",
			'11003' => "Algar",
			'11004' => "Algeciras",
			'11005' => "Algodonales",
			'11006' => "Arcos de la Frontera",
			'11007' => "Barbate",
			'11008' => "Barrios, Los",
			'11901' => "Benalup-Casas Viejas",
			'11009' => "Benaocaz",
			'11010' => "Bornos",
			'11011' => "Bosque, El",
			'11012' => "Cádiz",
			'11013' => "Castellar de la Frontera",
			'11015' => "Chiclana de la Frontera",
			'11016' => "Chipiona",
			'11014' => "Conil de la Frontera",
			'11017' => "Espera",
			'11018' => "Gastor, El",
			'11019' => "Grazalema",
			'11020' => "Jerez de la Frontera",
			'11021' => "Jimena de la Frontera",
			'11022' => "Línea de la Concepción, La",
			'11023' => "Medina Sidonia",
			'11024' => "Olvera",
			'11025' => "Paterna de Rivera",
			'11026' => "Prado del Rey",
			'11027' => "Puerto de Santa María, El",
			'11028' => "Puerto Real",
			'11029' => "Puerto Serrano",
			'11030' => "Rota",
			'11031' => "San Fernando",
			'11902' => "San José del Valle",
			'11903' => "San Martín del Tesorillo",
			'11033' => "San Roque",
			'11032' => "Sanlúcar de Barrameda",
			'11034' => "Setenil de las Bodegas",
			'11035' => "Tarifa",
			'11036' => "Torre Alháquime",
			'11037' => "Trebujena",
			'11038' => "Ubrique",
			'11039' => "Vejer de la Frontera",
			'11040' => "Villaluenga del Rosario",
			'11041' => "Villamartín",
			'11042' => "Zahara",
			'12002' => "Aín",
			'12003' => "Albocàsser",
			'12004' => "Alcalà de Xivert",
			'12005' => "Alcora, l'",
			'12006' => "Alcudia de Veo",
			'12007' => "Alfondeguilla",
			'12008' => "Algimia de Almonacid",
			'12009' => "Almassora",
			'12010' => "Almedíjar",
			'12011' => "Almenara",
			'12901' => "Alqueries, les/Alquerías del Niño Perdido",
			'12012' => "Altura",
			'12013' => "Arañuel",
			'12014' => "Ares del Maestrat",
			'12015' => "Argelita",
			'12016' => "Artana",
			'12001' => "Atzeneta del Maestrat",
			'12017' => "Ayódar",
			'12018' => "Azuébar",
			'12020' => "Barracas",
			'12022' => "Bejís",
			'12024' => "Benafer",
			'12025' => "Benafigos",
			'12026' => "Benassal",
			'12027' => "Benicarló",
			'12028' => "Benicàssim/Benicasim",
			'12029' => "Benlloc",
			'12021' => "Betxí",
			'12032' => "Borriana/Burriana",
			'12031' => "Borriol",
			'12033' => "Cabanes",
			'12034' => "Càlig",
			'12036' => "Canet lo Roig",
			'12037' => "Castell de Cabres",
			'12038' => "Castellfort",
			'12039' => "Castellnovo",
			'12040' => "Castelló de la Plana",
			'12041' => "Castillo de Villamalefa",
			'12042' => "Catí",
			'12043' => "Caudiel",
			'12044' => "Cervera del Maestre",
			'12053' => "Chilches/Xilxes",
			'12056' => "Chóvar",
			'12045' => "Cinctorres",
			'12046' => "Cirat",
			'12048' => "Cortes de Arenoso",
			'12049' => "Costur",
			'12050' => "Coves de Vinromà, les",
			'12051' => "Culla",
			'12057' => "Eslida",
			'12058' => "Espadilla",
			'12059' => "Fanzara",
			'12060' => "Figueroles",
			'12061' => "Forcall",
			'12063' => "Fuente la Reina",
			'12064' => "Fuentes de Ayódar",
			'12065' => "Gaibiel",
			'12067' => "Geldo",
			'12068' => "Herbers",
			'12069' => "Higueras",
			'12070' => "Jana, la",
			'12071' => "Jérica",
			'12074' => "Llosa, la",
			'12072' => "Llucena/Lucena del Cid",
			'12073' => "Ludiente",
			'12075' => "Mata de Morella, la",
			'12076' => "Matet",
			'12077' => "Moncofa",
			'12078' => "Montán",
			'12079' => "Montanejos",
			'12080' => "Morella",
			'12081' => "Navajas",
			'12082' => "Nules",
			'12083' => "Olocau del Rey",
			'12084' => "Onda",
			'12085' => "Orpesa/Oropesa del Mar",
			'12087' => "Palanques",
			'12088' => "Pavías",
			'12089' => "Peníscola/Peñíscola",
			'12090' => "Pina de Montalgrao",
			'12093' => "Pobla de Benifassà, la",
			'12094' => "Pobla Tornesa, la",
			'12091' => "Portell de Morella",
			'12092' => "Puebla de Arenoso",
			'12095' => "Ribesalbes",
			'12096' => "Rossell",
			'12097' => "Sacañet",
			'12098' => "Salzadella, la",
			'12101' => "San Rafael del Río",
			'12902' => "Sant Joan de Moró",
			'12099' => "Sant Jordi/San Jorge",
			'12100' => "Sant Mateu",
			'12102' => "Santa Magdalena de Pulpis",
			'12104' => "Segorbe",
			'12103' => "Serratella, la",
			'12105' => "Sierra Engarcerán",
			'12106' => "Soneja",
			'12107' => "Sot de Ferrer",
			'12108' => "Suera/Sueras",
			'12109' => "Tales",
			'12110' => "Teresa",
			'12111' => "Tírig",
			'12112' => "Todolella",
			'12113' => "Toga",
			'12114' => "Torás",
			'12115' => "Toro, El",
			'12116' => "Torralba del Pinar",
			'12119' => "Torre d'en Besora, la",
			'12120' => "Torre d'en Doménec, la",
			'12117' => "Torreblanca",
			'12118' => "Torrechiva",
			'12121' => "Traiguera",
			'12122' => "Useres, les/Useras",
			'12124' => "Vall d'Alba",
			'12125' => "Vall de Almonacid",
			'12126' => "Vall d'Uixó, la",
			'12123' => "Vallat",
			'12127' => "Vallibona",
			'12128' => "Vilafamés",
			'12129' => "Vilafranca/Villafranca del Cid",
			'12132' => "Vilanova d'Alcolea",
			'12134' => "Vilar de Canes",
			'12135' => "Vila-real",
			'12136' => "Vilavella, la",
			'12130' => "Villahermosa del Río",
			'12131' => "Villamalur",
			'12133' => "Villanueva de Viver",
			'12137' => "Villores",
			'12138' => "Vinaròs",
			'12139' => "Vistabella del Maestrat",
			'12140' => "Viver",
			'12052' => "Xert",
			'12055' => "Xodos/Chodos",
			'12141' => "Zorita del Maestrazgo",
			'12142' => "Zucaina",
			'13001' => "Abenójar",
			'13002' => "Agudo",
			'13003' => "Alamillo",
			'13004' => "Albaladejo",
			'13005' => "Alcázar de San Juan",
			'13006' => "Alcoba",
			'13007' => "Alcolea de Calatrava",
			'13008' => "Alcubillas",
			'13009' => "Aldea del Rey",
			'13010' => "Alhambra",
			'13011' => "Almadén",
			'13012' => "Almadenejos",
			'13013' => "Almagro",
			'13014' => "Almedina",
			'13015' => "Almodóvar del Campo",
			'13016' => "Almuradiel",
			'13017' => "Anchuras",
			'13903' => "Arenales de San Gregorio",
			'13018' => "Arenas de San Juan",
			'13019' => "Argamasilla de Alba",
			'13020' => "Argamasilla de Calatrava",
			'13021' => "Arroba de los Montes",
			'13022' => "Ballesteros de Calatrava",
			'13023' => "Bolaños de Calatrava",
			'13024' => "Brazatortas",
			'13025' => "Cabezarados",
			'13026' => "Cabezarrubias del Puerto",
			'13027' => "Calzada de Calatrava",
			'13028' => "Campo de Criptana",
			'13029' => "Cañada de Calatrava",
			'13030' => "Caracuel de Calatrava",
			'13031' => "Carrión de Calatrava",
			'13032' => "Carrizosa",
			'13033' => "Castellar de Santiago",
			'13038' => "Chillón",
			'13034' => "Ciudad Real",
			'13035' => "Corral de Calatrava",
			'13036' => "Cortijos, Los",
			'13037' => "Cózar",
			'13039' => "Daimiel",
			'13040' => "Fernán Caballero",
			'13041' => "Fontanarejo",
			'13042' => "Fuencaliente",
			'13043' => "Fuenllana",
			'13044' => "Fuente el Fresno",
			'13045' => "Granátula de Calatrava",
			'13046' => "Guadalmez",
			'13047' => "Herencia",
			'13048' => "Hinojosas de Calatrava",
			'13049' => "Horcajo de los Montes",
			'13050' => "Labores, Las",
			'13904' => "Llanos del Caudillo",
			'13051' => "Luciana",
			'13052' => "Malagón",
			'13053' => "Manzanares",
			'13054' => "Membrilla",
			'13055' => "Mestanza",
			'13056' => "Miguelturra",
			'13057' => "Montiel",
			'13058' => "Moral de Calatrava",
			'13059' => "Navalpino",
			'13060' => "Navas de Estena",
			'13061' => "Pedro Muñoz",
			'13062' => "Picón",
			'13063' => "Piedrabuena",
			'13064' => "Poblete",
			'13065' => "Porzuna",
			'13066' => "Pozuelo de Calatrava",
			'13067' => "Pozuelos de Calatrava, Los",
			'13068' => "Puebla de Don Rodrigo",
			'13069' => "Puebla del Príncipe",
			'13070' => "Puerto Lápice",
			'13071' => "Puertollano",
			'13072' => "Retuerta del Bullaque",
			'13901' => "Robledo, El",
			'13902' => "Ruidera",
			'13073' => "Saceruela",
			'13074' => "San Carlos del Valle",
			'13075' => "San Lorenzo de Calatrava",
			'13076' => "Santa Cruz de los Cáñamos",
			'13077' => "Santa Cruz de Mudela",
			'13078' => "Socuéllamos",
			'13080' => "Solana del Pino",
			'13079' => "Solana, La",
			'13081' => "Terrinches",
			'13082' => "Tomelloso",
			'13083' => "Torralba de Calatrava",
			'13084' => "Torre de Juan Abad",
			'13085' => "Torrenueva",
			'13086' => "Valdemanco del Esteras",
			'13087' => "Valdepeñas",
			'13088' => "Valenzuela de Calatrava",
			'13089' => "Villahermosa",
			'13090' => "Villamanrique",
			'13091' => "Villamayor de Calatrava",
			'13092' => "Villanueva de la Fuente",
			'13093' => "Villanueva de los Infantes",
			'13094' => "Villanueva de San Carlos",
			'13095' => "Villar del Pozo",
			'13096' => "Villarrubia de los Ojos",
			'13097' => "Villarta de San Juan",
			'13098' => "Viso del Marqués",
			'14001' => "Adamuz",
			'14002' => "Aguilar de la Frontera",
			'14003' => "Alcaracejos",
			'14004' => "Almedinilla",
			'14005' => "Almodóvar del Río",
			'14006' => "Añora",
			'14007' => "Baena",
			'14008' => "Belalcázar",
			'14009' => "Belmez",
			'14010' => "Benamejí",
			'14011' => "Blázquez, Los",
			'14012' => "Bujalance",
			'14013' => "Cabra",
			'14014' => "Cañete de las Torres",
			'14015' => "Carcabuey",
			'14016' => "Cardeña",
			'14017' => "Carlota, La",
			'14018' => "Carpio, El",
			'14019' => "Castro del Río",
			'14020' => "Conquista",
			'14021' => "Córdoba",
			'14022' => "Doña Mencía",
			'14023' => "Dos Torres",
			'14024' => "Encinas Reales",
			'14025' => "Espejo",
			'14026' => "Espiel",
			'14027' => "Fernán-Núñez",
			'14901' => "Fuente Carreteros",
			'14028' => "Fuente la Lancha",
			'14029' => "Fuente Obejuna",
			'14030' => "Fuente Palmera",
			'14031' => "Fuente-Tójar",
			'14032' => "Granjuela, La",
			'14033' => "Guadalcázar",
			'14902' => "Guijarrosa, La",
			'14034' => "Guijo, El",
			'14035' => "Hinojosa del Duque",
			'14036' => "Hornachuelos",
			'14037' => "Iznájar",
			'14038' => "Lucena",
			'14039' => "Luque",
			'14040' => "Montalbán de Córdoba",
			'14041' => "Montemayor",
			'14042' => "Montilla",
			'14043' => "Montoro",
			'14044' => "Monturque",
			'14045' => "Moriles",
			'14046' => "Nueva Carteya",
			'14047' => "Obejo",
			'14048' => "Palenciana",
			'14049' => "Palma del Río",
			'14050' => "Pedro Abad",
			'14051' => "Pedroche",
			'14052' => "Peñarroya-Pueblonuevo",
			'14053' => "Posadas",
			'14054' => "Pozoblanco",
			'14055' => "Priego de Córdoba",
			'14056' => "Puente Genil",
			'14057' => "Rambla, La",
			'14058' => "Rute",
			'14059' => "San Sebastián de los Ballesteros",
			'14061' => "Santa Eufemia",
			'14060' => "Santaella",
			'14062' => "Torrecampo",
			'14063' => "Valenzuela",
			'14064' => "Valsequillo",
			'14065' => "Victoria, La",
			'14066' => "Villa del Río",
			'14067' => "Villafranca de Córdoba",
			'14068' => "Villaharta",
			'14069' => "Villanueva de Córdoba",
			'14070' => "Villanueva del Duque",
			'14071' => "Villanueva del Rey",
			'14072' => "Villaralto",
			'14073' => "Villaviciosa de Córdoba",
			'14074' => "Viso, El",
			'14075' => "Zuheros",
			'15001' => "Abegondo",
			'15002' => "Ames",
			'15003' => "Aranga",
			'15004' => "Ares",
			'15005' => "Arteixo",
			'15006' => "Arzúa",
			'15007' => "Baña, A",
			'15008' => "Bergondo",
			'15009' => "Betanzos",
			'15010' => "Boimorto",
			'15011' => "Boiro",
			'15012' => "Boqueixón",
			'15013' => "Brión",
			'15014' => "Cabana de Bergantiños",
			'15015' => "Cabanas",
			'15016' => "Camariñas",
			'15017' => "Cambre",
			'15018' => "Capela, A",
			'15019' => "Carballo",
			'15901' => "Cariño",
			'15020' => "Carnota",
			'15021' => "Carral",
			'15022' => "Cedeira",
			'15023' => "Cee",
			'15024' => "Cerceda",
			'15025' => "Cerdido",
			'15027' => "Coirós",
			'15028' => "Corcubión",
			'15029' => "Coristanco",
			'15030' => "Coruña, A",
			'15031' => "Culleredo",
			'15032' => "Curtis",
			'15033' => "Dodro",
			'15034' => "Dumbría",
			'15035' => "Fene",
			'15036' => "Ferrol",
			'15037' => "Fisterra",
			'15038' => "Frades",
			'15039' => "Irixoa",
			'15041' => "Laracha, A",
			'15040' => "Laxe",
			'15042' => "Lousame",
			'15043' => "Malpica de Bergantiños",
			'15044' => "Mañón",
			'15045' => "Mazaricos",
			'15046' => "Melide",
			'15047' => "Mesía",
			'15048' => "Miño",
			'15049' => "Moeche",
			'15050' => "Monfero",
			'15051' => "Mugardos",
			'15053' => "Muros",
			'15052' => "Muxía",
			'15054' => "Narón",
			'15055' => "Neda",
			'15056' => "Negreira",
			'15057' => "Noia",
			'15058' => "Oleiros",
			'15059' => "Ordes",
			'15060' => "Oroso",
			'15061' => "Ortigueira",
			'15062' => "Outes",
			'15902' => "Oza-Cesuras",
			'15064' => "Paderne",
			'15065' => "Padrón",
			'15066' => "Pino, O",
			'15067' => "Pobra do Caramiñal, A",
			'15068' => "Ponteceso",
			'15069' => "Pontedeume",
			'15070' => "Pontes de García Rodríguez, As",
			'15071' => "Porto do Son",
			'15072' => "Rianxo",
			'15073' => "Ribeira",
			'15074' => "Rois",
			'15075' => "Sada",
			'15076' => "San Sadurniño",
			'15077' => "Santa Comba",
			'15078' => "Santiago de Compostela",
			'15079' => "Santiso",
			'15080' => "Sobrado",
			'15081' => "Somozas, As",
			'15082' => "Teo",
			'15083' => "Toques",
			'15084' => "Tordoia",
			'15085' => "Touro",
			'15086' => "Trazo",
			'15088' => "Val do Dubra",
			'15087' => "Valdoviño",
			'15089' => "Vedra",
			'15091' => "Vilarmaior",
			'15090' => "Vilasantar",
			'15092' => "Vimianzo",
			'15093' => "Zas",
			'16001' => "Abia de la Obispalía",
			'16002' => "Acebrón, El",
			'16003' => "Alarcón",
			'16004' => "Albaladejo del Cuende",
			'16005' => "Albalate de las Nogueras",
			'16006' => "Albendea",
			'16007' => "Alberca de Záncara, La",
			'16008' => "Alcalá de la Vega",
			'16009' => "Alcantud",
			'16010' => "Alcázar del Rey",
			'16011' => "Alcohujate",
			'16012' => "Alconchel de la Estrella",
			'16013' => "Algarra",
			'16014' => "Aliaguilla",
			'16015' => "Almarcha, La",
			'16016' => "Almendros",
			'16017' => "Almodóvar del Pinar",
			'16018' => "Almonacid del Marquesado",
			'16019' => "Altarejos",
			'16020' => "Arandilla del Arroyo",
			'16905' => "Arcas",
			'16022' => "Arcos de la Sierra",
			'16024' => "Arguisuelas",
			'16025' => "Arrancacepas",
			'16026' => "Atalaya del Cañavate",
			'16027' => "Barajas de Melo",
			'16029' => "Barchín del Hoyo",
			'16030' => "Bascuñana de San Pedro",
			'16031' => "Beamud",
			'16032' => "Belinchón",
			'16033' => "Belmonte",
			'16034' => "Belmontejo",
			'16035' => "Beteta",
			'16036' => "Boniches",
			'16038' => "Buciegas",
			'16039' => "Buenache de Alarcón",
			'16040' => "Buenache de la Sierra",
			'16041' => "Buendía",
			'16042' => "Campillo de Altobuey",
			'16043' => "Campillos-Paravientos",
			'16044' => "Campillos-Sierra",
			'16901' => "Campos del Paraíso",
			'16045' => "Canalejas del Arroyo",
			'16046' => "Cañada del Hoyo",
			'16047' => "Cañada Juncosa",
			'16048' => "Cañamares",
			'16049' => "Cañavate, El",
			'16050' => "Cañaveras",
			'16051' => "Cañaveruelas",
			'16052' => "Cañete",
			'16053' => "Cañizares",
			'16055' => "Carboneras de Guadazaón",
			'16056' => "Cardenete",
			'16057' => "Carrascosa",
			'16058' => "Carrascosa de Haro",
			'16060' => "Casas de Benítez",
			'16061' => "Casas de Fernando Alonso",
			'16062' => "Casas de Garcimolina",
			'16063' => "Casas de Guijarro",
			'16064' => "Casas de Haro",
			'16065' => "Casas de los Pinos",
			'16066' => "Casasimarro",
			'16067' => "Castejón",
			'16068' => "Castillejo de Iniesta",
			'16070' => "Castillejo-Sierra",
			'16072' => "Castillo de Garcimuñoz",
			'16071' => "Castillo-Albaráñez",
			'16073' => "Cervera del Llano",
			'16023' => "Chillarón de Cuenca",
			'16081' => "Chumillas",
			'16074' => "Cierva, La",
			'16078' => "Cuenca",
			'16079' => "Cueva del Hierro",
			'16082' => "Enguídanos",
			'16083' => "Fresneda de Altarejos",
			'16084' => "Fresneda de la Sierra",
			'16085' => "Frontera, La",
			'16086' => "Fuente de Pedro Naharro",
			'16087' => "Fuentelespino de Haro",
			'16088' => "Fuentelespino de Moya",
			'16904' => "Fuentenava de Jábaga",
			'16089' => "Fuentes",
			'16091' => "Fuertescusa",
			'16092' => "Gabaldón",
			'16093' => "Garaballa",
			'16094' => "Gascueña",
			'16095' => "Graja de Campalbo",
			'16096' => "Graja de Iniesta",
			'16097' => "Henarejos",
			'16098' => "Herrumblar, El",
			'16099' => "Hinojosa, La",
			'16100' => "Hinojosos, Los",
			'16101' => "Hito, El",
			'16102' => "Honrubia",
			'16103' => "Hontanaya",
			'16104' => "Hontecillas",
			'16106' => "Horcajo de Santiago",
			'16107' => "Huélamo",
			'16108' => "Huelves",
			'16109' => "Huérguina",
			'16110' => "Huerta de la Obispalía",
			'16111' => "Huerta del Marquesado",
			'16112' => "Huete",
			'16113' => "Iniesta",
			'16115' => "Laguna del Marquesado",
			'16116' => "Lagunaseca",
			'16117' => "Landete",
			'16118' => "Ledaña",
			'16119' => "Leganiel",
			'16121' => "Majadas, Las",
			'16122' => "Mariana",
			'16123' => "Masegosa",
			'16124' => "Mesas, Las",
			'16125' => "Minglanilla",
			'16126' => "Mira",
			'16128' => "Monreal del Llano",
			'16129' => "Montalbanejo",
			'16130' => "Montalbo",
			'16131' => "Monteagudo de las Salinas",
			'16132' => "Mota de Altarejos",
			'16133' => "Mota del Cuervo",
			'16134' => "Motilla del Palancar",
			'16135' => "Moya",
			'16137' => "Narboneta",
			'16139' => "Olivares de Júcar",
			'16140' => "Olmeda de la Cuesta",
			'16141' => "Olmeda del Rey",
			'16142' => "Olmedilla de Alarcón",
			'16143' => "Olmedilla de Eliz",
			'16145' => "Osa de la Vega",
			'16146' => "Pajarón",
			'16147' => "Pajaroncillo",
			'16148' => "Palomares del Campo",
			'16149' => "Palomera",
			'16150' => "Paracuellos",
			'16151' => "Paredes",
			'16152' => "Parra de las Vegas, La",
			'16153' => "Pedernoso, El",
			'16154' => "Pedroñeras, Las",
			'16155' => "Peral, El",
			'16156' => "Peraleja, La",
			'16157' => "Pesquera, La",
			'16158' => "Picazo, El",
			'16159' => "Pinarejo",
			'16160' => "Pineda de Gigüela",
			'16161' => "Piqueras del Castillo",
			'16162' => "Portalrubio de Guadamejud",
			'16163' => "Portilla",
			'16165' => "Poyatos",
			'16166' => "Pozoamargo",
			'16908' => "Pozorrubielos de la Mancha",
			'16167' => "Pozorrubio de Santiago",
			'16169' => "Pozuelo, El",
			'16170' => "Priego",
			'16171' => "Provencio, El",
			'16172' => "Puebla de Almenara",
			'16174' => "Puebla del Salvador",
			'16175' => "Quintanar del Rey",
			'16176' => "Rada de Haro",
			'16177' => "Reíllo",
			'16181' => "Rozalén del Monte",
			'16185' => "Saceda-Trasierra",
			'16186' => "Saelices",
			'16187' => "Salinas del Manzano",
			'16188' => "Salmeroncillos",
			'16189' => "Salvacañete",
			'16190' => "San Clemente",
			'16191' => "San Lorenzo de la Parrilla",
			'16192' => "San Martín de Boniches",
			'16193' => "San Pedro Palmiches",
			'16194' => "Santa Cruz de Moya",
			'16196' => "Santa María de los Llanos",
			'16195' => "Santa María del Campo Rus",
			'16197' => "Santa María del Val",
			'16198' => "Sisante",
			'16199' => "Solera de Gabaldón",
			'16909' => "Sotorribas",
			'16202' => "Talayuelas",
			'16203' => "Tarancón",
			'16204' => "Tébar",
			'16205' => "Tejadillos",
			'16206' => "Tinajas",
			'16209' => "Torralba",
			'16211' => "Torrejoncillo del Rey",
			'16212' => "Torrubia del Campo",
			'16213' => "Torrubia del Castillo",
			'16215' => "Tragacete",
			'16216' => "Tresjuncos",
			'16217' => "Tribaldos",
			'16218' => "Uclés",
			'16219' => "Uña",
			'16906' => "Valdecolmenas, Los",
			'16224' => "Valdemeca",
			'16225' => "Valdemorillo de la Sierra",
			'16227' => "Valdemoro-Sierra",
			'16228' => "Valdeolivas",
			'16902' => "Valdetórtola",
			'16903' => "Valeras, Las",
			'16231' => "Valhermoso de la Fuente",
			'16173' => "Valle de Altomira, El",
			'16234' => "Valsalobre",
			'16236' => "Valverde de Júcar",
			'16237' => "Valverdejo",
			'16238' => "Vara de Rey",
			'16239' => "Vega del Codorno",
			'16240' => "Vellisca",
			'16242' => "Villaconejos de Trabaque",
			'16243' => "Villaescusa de Haro",
			'16244' => "Villagarcía del Llano",
			'16245' => "Villalba de la Sierra",
			'16246' => "Villalba del Rey",
			'16247' => "Villalgordo del Marquesado",
			'16248' => "Villalpardo",
			'16249' => "Villamayor de Santiago",
			'16250' => "Villanueva de Guadamejud",
			'16251' => "Villanueva de la Jara",
			'16253' => "Villar de Cañas",
			'16254' => "Villar de Domingo García",
			'16255' => "Villar de la Encina",
			'16263' => "Villar de Olalla",
			'16258' => "Villar del Humo",
			'16259' => "Villar del Infantado",
			'16910' => "Villar y Velasco",
			'16264' => "Villarejo de Fuentes",
			'16265' => "Villarejo de la Peñuela",
			'16266' => "Villarejo-Periesteban",
			'16269' => "Villares del Saz",
			'16270' => "Villarrubio",
			'16271' => "Villarta",
			'16272' => "Villas de la Ventosa",
			'16273' => "Villaverde y Pasaconsol",
			'16274' => "Víllora",
			'16275' => "Vindel",
			'16276' => "Yémeda",
			'16277' => "Zafra de Záncara",
			'16278' => "Zafrilla",
			'16279' => "Zarza de Tajo",
			'16280' => "Zarzuela",
			'17001' => "Agullana",
			'17002' => "Aiguaviva",
			'17003' => "Albanyà",
			'17004' => "Albons",
			'17006' => "Alp",
			'17007' => "Amer",
			'17008' => "Anglès",
			'17009' => "Arbúcies",
			'17010' => "Argelaguer",
			'17011' => "Armentera, L'",
			'17012' => "Avinyonet de Puigventós",
			'17015' => "Banyoles",
			'17016' => "Bàscara",
			'17013' => "Begur",
			'17018' => "Bellcaire d'Empordà",
			'17019' => "Besalú",
			'17020' => "Bescanó",
			'17021' => "Beuda",
			'17022' => "Bisbal d'Empordà, La",
			'17234' => "Biure",
			'17023' => "Blanes",
			'17029' => "Boadella i les Escaules",
			'17024' => "Bolvir",
			'17025' => "Bordils",
			'17026' => "Borrassà",
			'17027' => "Breda",
			'17028' => "Brunyola i Sant Martí Sapresa",
			'17031' => "Cabanelles",
			'17030' => "Cabanes",
			'17032' => "Cadaqués",
			'17033' => "Caldes de Malavella",
			'17034' => "Calonge i Sant Antoni",
			'17035' => "Camós",
			'17036' => "Campdevànol",
			'17037' => "Campelles",
			'17038' => "Campllong",
			'17039' => "Camprodon",
			'17040' => "Canet d'Adri",
			'17041' => "Cantallops",
			'17042' => "Capmany",
			'17044' => "Cassà de la Selva",
			'17048' => "Castell d'Aro, Platja d'Aro i s'Agaró",
			'17046' => "Castellfollit de la Roca",
			'17047' => "Castelló d'Empúries",
			'17189' => "Cellera de Ter, La",
			'17049' => "Celrà",
			'17050' => "Cervià de Ter",
			'17051' => "Cistella",
			'17054' => "Colera",
			'17055' => "Colomers",
			'17057' => "Corçà",
			'17056' => "Cornellà del Terri",
			'17058' => "Crespià",
			'17901' => "Cruïlles, Monells i Sant Sadurní de l'Heura",
			'17060' => "Darnius",
			'17061' => "Das",
			'17062' => "Escala, L'",
			'17063' => "Espinelves",
			'17064' => "Espolla",
			'17065' => "Esponellà",
			'17005' => "Far d'Empordà, El",
			'17066' => "Figueres",
			'17067' => "Flaçà",
			'17068' => "Foixà",
			'17069' => "Fontanals de Cerdanya",
			'17070' => "Fontanilles",
			'17071' => "Fontcoberta",
			'17902' => "Forallac",
			'17073' => "Fornells de la Selva",
			'17074' => "Fortià",
			'17075' => "Garrigàs",
			'17076' => "Garrigoles",
			'17077' => "Garriguella",
			'17078' => "Ger",
			'17079' => "Girona",
			'17080' => "Gombrèn",
			'17081' => "Gualta",
			'17082' => "Guils de Cerdanya",
			'17083' => "Hostalric",
			'17084' => "Isòvol",
			'17085' => "Jafre",
			'17086' => "Jonquera, La",
			'17087' => "Juià",
			'17088' => "Lladó",
			'17089' => "Llagostera",
			'17090' => "Llambilles",
			'17091' => "Llanars",
			'17092' => "Llançà",
			'17093' => "Llers",
			'17094' => "Llívia",
			'17095' => "Lloret de Mar",
			'17096' => "Llosses, Les",
			'17102' => "Maçanet de Cabrenys",
			'17103' => "Maçanet de la Selva",
			'17097' => "Madremanya",
			'17098' => "Maià de Montcal",
			'17100' => "Masarac",
			'17101' => "Massanes",
			'17099' => "Meranges",
			'17105' => "Mieres",
			'17106' => "Mollet de Peralada",
			'17107' => "Molló",
			'17109' => "Montagut i Oix",
			'17110' => "Mont-ras",
			'17111' => "Navata",
			'17112' => "Ogassa",
			'17114' => "Olot",
			'17115' => "Ordis",
			'17116' => "Osor",
			'17117' => "Palafrugell",
			'17118' => "Palamós",
			'17119' => "Palau de Santa Eulàlia",
			'17121' => "Palau-sator",
			'17120' => "Palau-saverdera",
			'17123' => "Palol de Revardit",
			'17124' => "Pals",
			'17125' => "Pardines",
			'17126' => "Parlavà",
			'17128' => "Pau",
			'17129' => "Pedret i Marzà",
			'17130' => "Pera, La",
			'17132' => "Peralada",
			'17133' => "Planes d'Hostoles, Les",
			'17134' => "Planoles",
			'17135' => "Pont de Molins",
			'17136' => "Pontós",
			'17137' => "Porqueres",
			'17140' => "Port de la Selva, El",
			'17138' => "Portbou",
			'17139' => "Preses, Les",
			'17141' => "Puigcerdà",
			'17142' => "Quart",
			'17043' => "Queralbs",
			'17143' => "Rabós",
			'17144' => "Regencós",
			'17145' => "Ribes de Freser",
			'17146' => "Riells i Viabrea",
			'17147' => "Ripoll",
			'17148' => "Riudarenes",
			'17149' => "Riudaura",
			'17150' => "Riudellots de la Selva",
			'17151' => "Riumors",
			'17152' => "Roses",
			'17153' => "Rupià",
			'17154' => "Sales de Llierca",
			'17155' => "Salt",
			'17157' => "Sant Andreu Salou",
			'17183' => "Sant Aniol de Finestres",
			'17158' => "Sant Climent Sescebes",
			'17159' => "Sant Feliu de Buixalleu",
			'17160' => "Sant Feliu de Guíxols",
			'17161' => "Sant Feliu de Pallerols",
			'17162' => "Sant Ferriol",
			'17163' => "Sant Gregori",
			'17164' => "Sant Hilari Sacalm",
			'17165' => "Sant Jaume de Llierca",
			'17167' => "Sant Joan de les Abadesses",
			'17168' => "Sant Joan de Mollet",
			'17185' => "Sant Joan les Fonts",
			'17166' => "Sant Jordi Desvalls",
			'17169' => "Sant Julià de Ramis",
			'17903' => "Sant Julià del Llor i Bonmatí",
			'17171' => "Sant Llorenç de la Muga",
			'17172' => "Sant Martí de Llémena",
			'17173' => "Sant Martí Vell",
			'17174' => "Sant Miquel de Campmajor",
			'17175' => "Sant Miquel de Fluvià",
			'17176' => "Sant Mori",
			'17177' => "Sant Pau de Segúries",
			'17178' => "Sant Pere Pescador",
			'17180' => "Santa Coloma de Farners",
			'17181' => "Santa Cristina d'Aro",
			'17182' => "Santa Llogaia d'Àlguema",
			'17184' => "Santa Pau",
			'17186' => "Sarrià de Ter",
			'17187' => "Saus, Camallera i Llampaies",
			'17188' => "Selva de Mar, La",
			'17190' => "Serinyà",
			'17191' => "Serra de Daró",
			'17192' => "Setcases",
			'17193' => "Sils",
			'17052' => "Siurana",
			'17194' => "Susqueda",
			'17195' => "Tallada d'Empordà, La",
			'17196' => "Terrades",
			'17197' => "Torrent",
			'17198' => "Torroella de Fluvià",
			'17199' => "Torroella de Montgrí",
			'17200' => "Tortellà",
			'17201' => "Toses",
			'17202' => "Tossa de Mar",
			'17204' => "Ullà",
			'17205' => "Ullastret",
			'17203' => "Ultramort",
			'17206' => "Urús",
			'17014' => "Vajol, La",
			'17208' => "Vall de Bianya, La",
			'17207' => "Vall d'en Bas, La",
			'17170' => "Vallfogona de Ripollès",
			'17209' => "Vall-llobrega",
			'17210' => "Ventalló",
			'17211' => "Verges",
			'17212' => "Vidrà",
			'17213' => "Vidreres",
			'17214' => "Vilabertran",
			'17215' => "Vilablareix",
			'17217' => "Viladamat",
			'17216' => "Viladasens",
			'17218' => "Vilademuls",
			'17220' => "Viladrau",
			'17221' => "Vilafant",
			'17223' => "Vilajuïga",
			'17224' => "Vilallonga de Ter",
			'17225' => "Vilamacolum",
			'17226' => "Vilamalla",
			'17227' => "Vilamaniscle",
			'17228' => "Vilanant",
			'17230' => "Vila-sacra",
			'17222' => "Vilaür",
			'17233' => "Vilobí d'Onyar",
			'17232' => "Vilopriu",
			'18001' => "Agrón",
			'18002' => "Alamedilla",
			'18003' => "Albolote",
			'18004' => "Albondón",
			'18005' => "Albuñán",
			'18006' => "Albuñol",
			'18007' => "Albuñuelas",
			'18010' => "Aldeire",
			'18011' => "Alfacar",
			'18012' => "Algarinejo",
			'18013' => "Alhama de Granada",
			'18014' => "Alhendín",
			'18015' => "Alicún de Ortega",
			'18016' => "Almegíjar",
			'18017' => "Almuñécar",
			'18904' => "Alpujarra de la Sierra",
			'18018' => "Alquife",
			'18020' => "Arenas del Rey",
			'18021' => "Armilla",
			'18022' => "Atarfe",
			'18023' => "Baza",
			'18024' => "Beas de Granada",
			'18025' => "Beas de Guadix",
			'18027' => "Benalúa",
			'18028' => "Benalúa de las Villas",
			'18029' => "Benamaurel",
			'18030' => "Bérchules",
			'18032' => "Bubión",
			'18033' => "Busquístar",
			'18034' => "Cacín",
			'18035' => "Cádiar",
			'18036' => "Cájar",
			'18114' => "Calahorra, La",
			'18037' => "Calicasas",
			'18038' => "Campotéjar",
			'18039' => "Caniles",
			'18040' => "Cáñar",
			'18042' => "Capileira",
			'18043' => "Carataunas",
			'18044' => "Cástaras",
			'18045' => "Castilléjar",
			'18046' => "Castril",
			'18047' => "Cenes de la Vega",
			'18059' => "Chauchina",
			'18061' => "Chimeneas",
			'18062' => "Churriana de la Vega",
			'18048' => "Cijuela",
			'18049' => "Cogollos de Guadix",
			'18050' => "Cogollos de la Vega",
			'18051' => "Colomera",
			'18053' => "Cortes de Baza",
			'18054' => "Cortes y Graena",
			'18912' => "Cuevas del Campo",
			'18056' => "Cúllar",
			'18057' => "Cúllar Vega",
			'18063' => "Darro",
			'18064' => "Dehesas de Guadix",
			'18065' => "Dehesas Viejas",
			'18066' => "Deifontes",
			'18067' => "Diezma",
			'18068' => "Dílar",
			'18069' => "Dólar",
			'18915' => "Domingo Pérez de Granada",
			'18070' => "Dúdar",
			'18071' => "Dúrcal",
			'18072' => "Escúzar",
			'18074' => "Ferreira",
			'18076' => "Fonelas",
			'18077' => "Fornes",
			'18078' => "Freila",
			'18079' => "Fuente Vaqueros",
			'18905' => "Gabias, Las",
			'18082' => "Galera",
			'18083' => "Gobernador",
			'18084' => "Gójar",
			'18085' => "Gor",
			'18086' => "Gorafe",
			'18087' => "Granada",
			'18088' => "Guadahortuna",
			'18089' => "Guadix",
			'18906' => "Guájares, Los",
			'18093' => "Gualchos",
			'18094' => "Güéjar Sierra",
			'18095' => "Güevéjar",
			'18096' => "Huélago",
			'18097' => "Huéneja",
			'18098' => "Huéscar",
			'18099' => "Huétor de Santillán",
			'18100' => "Huétor Tájar",
			'18101' => "Huétor Vega",
			'18102' => "Íllora",
			'18103' => "Ítrabo",
			'18105' => "Iznalloz",
			'18106' => "Játar",
			'18107' => "Jayena",
			'18108' => "Jérez del Marquesado",
			'18109' => "Jete",
			'18111' => "Jun",
			'18112' => "Juviles",
			'18115' => "Láchar",
			'18116' => "Lanjarón",
			'18117' => "Lanteira",
			'18119' => "Lecrín",
			'18120' => "Lentegí",
			'18121' => "Lobras",
			'18122' => "Loja",
			'18123' => "Lugros",
			'18124' => "Lújar",
			'18126' => "Malahá, La",
			'18127' => "Maracena",
			'18128' => "Marchal",
			'18132' => "Moclín",
			'18133' => "Molvízar",
			'18134' => "Monachil",
			'18135' => "Montefrío",
			'18136' => "Montejícar",
			'18137' => "Montillana",
			'18138' => "Moraleda de Zafayona",
			'18909' => "Morelábor",
			'18140' => "Motril",
			'18141' => "Murtas",
			'18903' => "Nevada",
			'18143' => "Nigüelas",
			'18144' => "Nívar",
			'18145' => "Ogíjares",
			'18146' => "Orce",
			'18147' => "Órgiva",
			'18148' => "Otívar",
			'18150' => "Padul",
			'18151' => "Pampaneira",
			'18152' => "Pedro Martínez",
			'18153' => "Peligros",
			'18154' => "Peza, La",
			'18910' => "Pinar, El",
			'18157' => "Pinos Genil",
			'18158' => "Pinos Puente",
			'18159' => "Píñar",
			'18161' => "Polícar",
			'18162' => "Polopos",
			'18163' => "Pórtugos",
			'18164' => "Puebla de Don Fadrique",
			'18165' => "Pulianas",
			'18167' => "Purullena",
			'18168' => "Quéntar",
			'18170' => "Rubite",
			'18171' => "Salar",
			'18173' => "Salobreña",
			'18174' => "Santa Cruz del Comercio",
			'18175' => "Santa Fe",
			'18176' => "Soportújar",
			'18177' => "Sorvilán",
			'18901' => "Taha, La",
			'18178' => "Torre-Cardela",
			'18916' => "Torrenueva Costa",
			'18179' => "Torvizcón",
			'18180' => "Trevélez",
			'18181' => "Turón",
			'18182' => "Ugíjar",
			'18914' => "Valderrubio",
			'18907' => "Valle del Zalabí",
			'18902' => "Valle, El",
			'18183' => "Válor",
			'18911' => "Vegas del Genil",
			'18184' => "Vélez de Benaudalla",
			'18185' => "Ventas de Huelma",
			'18149' => "Villa de Otura",
			'18908' => "Villamena",
			'18187' => "Villanueva de las Torres",
			'18188' => "Villanueva Mesía",
			'18189' => "Víznar",
			'18192' => "Zafarraya",
			'18913' => "Zagra",
			'18193' => "Zubia, La",
			'18194' => "Zújar",
			'19001' => "Abánades",
			'19002' => "Ablanque",
			'19003' => "Adobes",
			'19004' => "Alaminos",
			'19005' => "Alarilla",
			'19006' => "Albalate de Zorita",
			'19007' => "Albares",
			'19008' => "Albendiego",
			'19009' => "Alcocer",
			'19010' => "Alcolea de las Peñas",
			'19011' => "Alcolea del Pinar",
			'19013' => "Alcoroches",
			'19015' => "Aldeanueva de Guadalajara",
			'19016' => "Algar de Mesa",
			'19017' => "Algora",
			'19018' => "Alhóndiga",
			'19019' => "Alique",
			'19020' => "Almadrones",
			'19021' => "Almoguera",
			'19022' => "Almonacid de Zorita",
			'19023' => "Alocén",
			'19024' => "Alovera",
			'19027' => "Alustante",
			'19031' => "Angón",
			'19032' => "Anguita",
			'19033' => "Anquela del Ducado",
			'19034' => "Anquela del Pedregal",
			'19036' => "Aranzueque",
			'19037' => "Arbancón",
			'19038' => "Arbeteta",
			'19039' => "Argecilla",
			'19040' => "Armallones",
			'19041' => "Armuña de Tajuña",
			'19042' => "Arroyo de las Fraguas",
			'19043' => "Atanzón",
			'19044' => "Atienza",
			'19045' => "Auñón",
			'19046' => "Azuqueca de Henares",
			'19047' => "Baides",
			'19048' => "Baños de Tajo",
			'19049' => "Bañuelos",
			'19050' => "Barriopedro",
			'19051' => "Berninches",
			'19052' => "Bodera, La",
			'19053' => "Brihuega",
			'19054' => "Budia",
			'19055' => "Bujalaro",
			'19057' => "Bustares",
			'19058' => "Cabanillas del Campo",
			'19059' => "Campillo de Dueñas",
			'19060' => "Campillo de Ranas",
			'19061' => "Campisábalos",
			'19064' => "Canredondo",
			'19065' => "Cantalojas",
			'19066' => "Cañizar",
			'19067' => "Cardoso de la Sierra, El",
			'19070' => "Casa de Uceda",
			'19071' => "Casar, El",
			'19073' => "Casas de San Galindo",
			'19074' => "Caspueñas",
			'19075' => "Castejón de Henares",
			'19076' => "Castellar de la Muela",
			'19078' => "Castilforte",
			'19079' => "Castilnuevo",
			'19080' => "Cendejas de Enmedio",
			'19081' => "Cendejas de la Torre",
			'19082' => "Centenera",
			'19103' => "Checa",
			'19104' => "Chequilla",
			'19106' => "Chillarón del Rey",
			'19105' => "Chiloeches",
			'19086' => "Cifuentes",
			'19087' => "Cincovillas",
			'19088' => "Ciruelas",
			'19089' => "Ciruelos del Pinar",
			'19090' => "Cobeta",
			'19091' => "Cogollor",
			'19092' => "Cogolludo",
			'19095' => "Condemios de Abajo",
			'19096' => "Condemios de Arriba",
			'19097' => "Congostrina",
			'19098' => "Copernal",
			'19099' => "Corduente",
			'19102' => "Cubillo de Uceda, El",
			'19107' => "Driebes",
			'19108' => "Durón",
			'19109' => "Embid",
			'19110' => "Escamilla",
			'19111' => "Escariche",
			'19112' => "Escopete",
			'19113' => "Espinosa de Henares",
			'19114' => "Esplegares",
			'19115' => "Establés",
			'19116' => "Estriégana",
			'19117' => "Fontanar",
			'19118' => "Fuembellida",
			'19119' => "Fuencemillán",
			'19120' => "Fuentelahiguera de Albatages",
			'19121' => "Fuentelencina",
			'19122' => "Fuentelsaz",
			'19123' => "Fuentelviejo",
			'19124' => "Fuentenovilla",
			'19125' => "Gajanejos",
			'19126' => "Galápagos",
			'19127' => "Galve de Sorbe",
			'19129' => "Gascueña de Bornova",
			'19130' => "Guadalajara",
			'19132' => "Henche",
			'19133' => "Heras de Ayuso",
			'19134' => "Herrería",
			'19135' => "Hiendelaencina",
			'19136' => "Hijes",
			'19138' => "Hita",
			'19139' => "Hombrados",
			'19142' => "Hontoba",
			'19143' => "Horche",
			'19145' => "Hortezuela de Océn",
			'19146' => "Huerce, La",
			'19147' => "Huérmeces del Cerro",
			'19148' => "Huertahernando",
			'19150' => "Hueva",
			'19151' => "Humanes",
			'19152' => "Illana",
			'19153' => "Iniéstola",
			'19154' => "Inviernas, Las",
			'19155' => "Irueste",
			'19156' => "Jadraque",
			'19157' => "Jirueque",
			'19159' => "Ledanca",
			'19160' => "Loranca de Tajuña",
			'19161' => "Lupiana",
			'19162' => "Luzaga",
			'19163' => "Luzón",
			'19165' => "Majaelrayo",
			'19166' => "Málaga del Fresno",
			'19167' => "Malaguilla",
			'19168' => "Mandayona",
			'19169' => "Mantiel",
			'19170' => "Maranchón",
			'19171' => "Marchamalo",
			'19172' => "Masegoso de Tajuña",
			'19173' => "Matarrubia",
			'19174' => "Matillas",
			'19175' => "Mazarete",
			'19176' => "Mazuecos",
			'19177' => "Medranda",
			'19178' => "Megina",
			'19179' => "Membrillera",
			'19181' => "Miedes de Atienza",
			'19182' => "Mierla, La",
			'19184' => "Millana",
			'19183' => "Milmarcos",
			'19185' => "Miñosa, La",
			'19186' => "Mirabueno",
			'19187' => "Miralrío",
			'19188' => "Mochales",
			'19189' => "Mohernando",
			'19190' => "Molina de Aragón",
			'19191' => "Monasterio",
			'19192' => "Mondéjar",
			'19193' => "Montarrón",
			'19194' => "Moratilla de los Meleros",
			'19195' => "Morenilla",
			'19196' => "Muduex",
			'19197' => "Navas de Jadraque, Las",
			'19198' => "Negredo",
			'19199' => "Ocentejo",
			'19200' => "Olivar, El",
			'19201' => "Olmeda de Cobeta",
			'19202' => "Olmeda de Jadraque, La",
			'19203' => "Ordial, El",
			'19204' => "Orea",
			'19208' => "Pálmaces de Jadraque",
			'19209' => "Pardos",
			'19210' => "Paredes de Sigüenza",
			'19211' => "Pareja",
			'19212' => "Pastrana",
			'19213' => "Pedregal, El",
			'19214' => "Peñalén",
			'19215' => "Peñalver",
			'19216' => "Peralejos de las Truchas",
			'19217' => "Peralveche",
			'19218' => "Pinilla de Jadraque",
			'19219' => "Pinilla de Molina",
			'19220' => "Pioz",
			'19221' => "Piqueras",
			'19222' => "Pobo de Dueñas, El",
			'19223' => "Poveda de la Sierra",
			'19224' => "Pozo de Almoguera",
			'19225' => "Pozo de Guadalajara",
			'19226' => "Prádena de Atienza",
			'19227' => "Prados Redondos",
			'19228' => "Puebla de Beleña",
			'19229' => "Puebla de Valles",
			'19230' => "Quer",
			'19231' => "Rebollosa de Jadraque",
			'19232' => "Recuenco, El",
			'19233' => "Renera",
			'19234' => "Retiendas",
			'19235' => "Riba de Saelices",
			'19237' => "Rillo de Gallo",
			'19238' => "Riofrío del Llano",
			'19239' => "Robledillo de Mohernando",
			'19240' => "Robledo de Corpes",
			'19241' => "Romanillos de Atienza",
			'19242' => "Romanones",
			'19243' => "Rueda de la Sierra",
			'19244' => "Sacecorbo",
			'19245' => "Sacedón",
			'19246' => "Saelices de la Sal",
			'19247' => "Salmerón",
			'19248' => "San Andrés del Congosto",
			'19249' => "San Andrés del Rey",
			'19250' => "Santiuste",
			'19251' => "Saúca",
			'19252' => "Sayatón",
			'19254' => "Selas",
			'19901' => "Semillas",
			'19255' => "Setiles",
			'19256' => "Sienes",
			'19257' => "Sigüenza",
			'19258' => "Solanillos del Extremo",
			'19259' => "Somolinos",
			'19260' => "Sotillo, El",
			'19261' => "Sotodosos",
			'19262' => "Tamajón",
			'19263' => "Taragudo",
			'19264' => "Taravilla",
			'19265' => "Tartanedo",
			'19266' => "Tendilla",
			'19267' => "Terzaga",
			'19268' => "Tierzo",
			'19269' => "Toba, La",
			'19271' => "Tordellego",
			'19270' => "Tordelrábano",
			'19272' => "Tordesilos",
			'19274' => "Torija",
			'19279' => "Torre del Burgo",
			'19277' => "Torrecuadrada de Molina",
			'19278' => "Torrecuadradilla",
			'19280' => "Torrejón del Rey",
			'19281' => "Torremocha de Jadraque",
			'19282' => "Torremocha del Campo",
			'19283' => "Torremocha del Pinar",
			'19284' => "Torremochuela",
			'19285' => "Torrubia",
			'19286' => "Tórtola de Henares",
			'19287' => "Tortuera",
			'19288' => "Tortuero",
			'19289' => "Traíd",
			'19290' => "Trijueque",
			'19291' => "Trillo",
			'19293' => "Uceda",
			'19294' => "Ujados",
			'19296' => "Utande",
			'19297' => "Valdarachas",
			'19298' => "Valdearenas",
			'19299' => "Valdeavellano",
			'19300' => "Valdeaveruelo",
			'19301' => "Valdeconcha",
			'19302' => "Valdegrudas",
			'19303' => "Valdelcubo",
			'19304' => "Valdenuño Fernández",
			'19305' => "Valdepeñas de la Sierra",
			'19306' => "Valderrebollo",
			'19307' => "Valdesotos",
			'19308' => "Valfermoso de Tajuña",
			'19309' => "Valhermoso",
			'19310' => "Valtablado del Río",
			'19311' => "Valverde de los Arroyos",
			'19314' => "Viana de Jadraque",
			'19317' => "Villanueva de Alcorón",
			'19318' => "Villanueva de Argecilla",
			'19319' => "Villanueva de la Torre",
			'19321' => "Villares de Jadraque",
			'19322' => "Villaseca de Henares",
			'19323' => "Villaseca de Uceda",
			'19324' => "Villel de Mesa",
			'19325' => "Viñuelas",
			'19326' => "Yebes",
			'19327' => "Yebra",
			'19329' => "Yélamos de Abajo",
			'19330' => "Yélamos de Arriba",
			'19331' => "Yunquera de Henares",
			'19332' => "Yunta, La",
			'19333' => "Zaorejas",
			'19334' => "Zarzuela de Jadraque",
			'19335' => "Zorita de los Canes",
			'20001' => "Abaltzisketa",
			'20002' => "Aduna",
			'20016' => "Aia",
			'20003' => "Aizarnazabal",
			'20004' => "Albiztur",
			'20005' => "Alegia",
			'20006' => "Alkiza",
			'20906' => "Altzaga",
			'20007' => "Altzo",
			'20008' => "Amezketa",
			'20009' => "Andoain",
			'20010' => "Anoeta",
			'20011' => "Antzuola",
			'20012' => "Arama",
			'20013' => "Aretxabaleta",
			'20055' => "Arrasate/Mondragón",
			'20014' => "Asteasu",
			'20903' => "Astigarraga",
			'20015' => "Ataun",
			'20017' => "Azkoitia",
			'20018' => "Azpeitia",
			'20904' => "Baliarrain",
			'20019' => "Beasain",
			'20020' => "Beizama",
			'20021' => "Belauntza",
			'20022' => "Berastegi",
			'20074' => "Bergara",
			'20023' => "Berrobi",
			'20024' => "Bidania-Goiatz",
			'20029' => "Deba",
			'20069' => "Donostia/San Sebastián",
			'20030' => "Eibar",
			'20031' => "Elduain",
			'20033' => "Elgeta",
			'20032' => "Elgoibar",
			'20067' => "Errenteria",
			'20066' => "Errezil",
			'20034' => "Eskoriatza",
			'20035' => "Ezkio-Itsaso",
			'20038' => "Gabiria",
			'20037' => "Gaintza",
			'20907' => "Gaztelu",
			'20039' => "Getaria",
			'20040' => "Hernani",
			'20041' => "Hernialde",
			'20036' => "Hondarribia",
			'20042' => "Ibarra",
			'20043' => "Idiazabal",
			'20044' => "Ikaztegieta",
			'20045' => "Irun",
			'20046' => "Irura",
			'20047' => "Itsasondo",
			'20048' => "Larraul",
			'20902' => "Lasarte-Oria",
			'20049' => "Lazkao",
			'20050' => "Leaburu",
			'20051' => "Legazpi",
			'20052' => "Legorreta",
			'20068' => "Leintz-Gatzaga",
			'20053' => "Lezo",
			'20054' => "Lizartza",
			'20901' => "Mendaro",
			'20057' => "Mutiloa",
			'20056' => "Mutriku",
			'20063' => "Oiartzun",
			'20058' => "Olaberria",
			'20059' => "Oñati",
			'20076' => "Ordizia",
			'20905' => "Orendain",
			'20060' => "Orexa",
			'20061' => "Orio",
			'20062' => "Ormaiztegi",
			'20064' => "Pasaia",
			'20070' => "Segura",
			'20065' => "Soraluze-Placencia de las Armas",
			'20071' => "Tolosa",
			'20072' => "Urnieta",
			'20077' => "Urretxu",
			'20073' => "Usurbil",
			'20075' => "Villabona",
			'20078' => "Zaldibia",
			'20079' => "Zarautz",
			'20025' => "Zegama",
			'20026' => "Zerain",
			'20027' => "Zestoa",
			'20028' => "Zizurkil",
			'20081' => "Zumaia",
			'20080' => "Zumarraga",
			'21001' => "Alájar",
			'21002' => "Aljaraque",
			'21003' => "Almendro, El",
			'21004' => "Almonaster la Real",
			'21005' => "Almonte",
			'21006' => "Alosno",
			'21007' => "Aracena",
			'21008' => "Aroche",
			'21009' => "Arroyomolinos de León",
			'21010' => "Ayamonte",
			'21011' => "Beas",
			'21012' => "Berrocal",
			'21013' => "Bollullos Par del Condado",
			'21014' => "Bonares",
			'21015' => "Cabezas Rubias",
			'21016' => "Cala",
			'21017' => "Calañas",
			'21018' => "Campillo, El",
			'21019' => "Campofrío",
			'21020' => "Cañaveral de León",
			'21021' => "Cartaya",
			'21022' => "Castaño del Robledo",
			'21023' => "Cerro de Andévalo, El",
			'21030' => "Chucena",
			'21024' => "Corteconcepción",
			'21025' => "Cortegana",
			'21026' => "Cortelazor",
			'21027' => "Cumbres de Enmedio",
			'21028' => "Cumbres de San Bartolomé",
			'21029' => "Cumbres Mayores",
			'21031' => "Encinasola",
			'21032' => "Escacena del Campo",
			'21033' => "Fuenteheridos",
			'21034' => "Galaroza",
			'21035' => "Gibraleón",
			'21036' => "Granada de Río-Tinto, La",
			'21037' => "Granado, El",
			'21038' => "Higuera de la Sierra",
			'21039' => "Hinojales",
			'21040' => "Hinojos",
			'21041' => "Huelva",
			'21042' => "Isla Cristina",
			'21043' => "Jabugo",
			'21044' => "Lepe",
			'21045' => "Linares de la Sierra",
			'21046' => "Lucena del Puerto",
			'21047' => "Manzanilla",
			'21048' => "Marines, Los",
			'21049' => "Minas de Riotinto",
			'21050' => "Moguer",
			'21051' => "Nava, La",
			'21052' => "Nerva",
			'21053' => "Niebla",
			'21054' => "Palma del Condado, La",
			'21055' => "Palos de la Frontera",
			'21056' => "Paterna del Campo",
			'21057' => "Paymogo",
			'21058' => "Puebla de Guzmán",
			'21059' => "Puerto Moral",
			'21060' => "Punta Umbría",
			'21061' => "Rociana del Condado",
			'21062' => "Rosal de la Frontera",
			'21063' => "San Bartolomé de la Torre",
			'21064' => "San Juan del Puerto",
			'21066' => "San Silvestre de Guzmán",
			'21065' => "Sanlúcar de Guadiana",
			'21067' => "Santa Ana la Real",
			'21068' => "Santa Bárbara de Casa",
			'21069' => "Santa Olalla del Cala",
			'21070' => "Trigueros",
			'21071' => "Valdelarco",
			'21072' => "Valverde del Camino",
			'21073' => "Villablanca",
			'21074' => "Villalba del Alcor",
			'21075' => "Villanueva de las Cruces",
			'21076' => "Villanueva de los Castillejos",
			'21077' => "Villarrasa",
			'21078' => "Zalamea la Real",
			'21902' => "Zarza-Perrunal, La",
			'21079' => "Zufre",
			'22001' => "Abiego",
			'22002' => "Abizanda",
			'22003' => "Adahuesca",
			'22004' => "Agüero",
			'22907' => "Aínsa-Sobrarbe",
			'22006' => "Aisa",
			'22007' => "Albalate de Cinca",
			'22008' => "Albalatillo",
			'22009' => "Albelda",
			'22011' => "Albero Alto",
			'22012' => "Albero Bajo",
			'22013' => "Alberuela de Tubo",
			'22014' => "Alcalá de Gurrea",
			'22015' => "Alcalá del Obispo",
			'22016' => "Alcampell",
			'22017' => "Alcolea de Cinca",
			'22018' => "Alcubierre",
			'22019' => "Alerre",
			'22020' => "Alfántega",
			'22021' => "Almudévar",
			'22022' => "Almunia de San Juan",
			'22023' => "Almuniente",
			'22024' => "Alquézar",
			'22025' => "Altorricón",
			'22027' => "Angüés",
			'22028' => "Ansó",
			'22029' => "Antillón",
			'22032' => "Aragüés del Puerto",
			'22035' => "Arén",
			'22036' => "Argavieso",
			'22037' => "Arguis",
			'22039' => "Ayerbe",
			'22040' => "Azanuy-Alins",
			'22041' => "Azara",
			'22042' => "Azlor",
			'22043' => "Baélls",
			'22044' => "Bailo",
			'22045' => "Baldellou",
			'22046' => "Ballobar",
			'22047' => "Banastás",
			'22048' => "Barbastro",
			'22049' => "Barbués",
			'22050' => "Barbuñales",
			'22051' => "Bárcabo",
			'22052' => "Belver de Cinca",
			'22053' => "Benabarre",
			'22054' => "Benasque",
			'22246' => "Beranuy",
			'22055' => "Berbegal",
			'22057' => "Bielsa",
			'22058' => "Bierge",
			'22059' => "Biescas",
			'22060' => "Binaced",
			'22061' => "Binéfar",
			'22062' => "Bisaurri",
			'22063' => "Biscarrués",
			'22064' => "Blecua y Torres",
			'22066' => "Boltaña",
			'22067' => "Bonansa",
			'22068' => "Borau",
			'22069' => "Broto",
			'22072' => "Caldearenas",
			'22074' => "Campo",
			'22075' => "Camporrélls",
			'22076' => "Canal de Berdún",
			'22077' => "Candasnos",
			'22078' => "Canfranc",
			'22079' => "Capdesaso",
			'22080' => "Capella",
			'22081' => "Casbas de Huesca",
			'22083' => "Castejón de Monegros",
			'22084' => "Castejón de Sos",
			'22082' => "Castejón del Puente",
			'22085' => "Castelflorite",
			'22086' => "Castiello de Jaca",
			'22087' => "Castigaleu",
			'22088' => "Castillazuelo",
			'22089' => "Castillonroy",
			'22094' => "Chalamera",
			'22095' => "Chía",
			'22096' => "Chimillas",
			'22090' => "Colungo",
			'22099' => "Esplús",
			'22102' => "Estada",
			'22103' => "Estadilla",
			'22105' => "Estopiñán del Castillo",
			'22106' => "Fago",
			'22107' => "Fanlo",
			'22109' => "Fiscal",
			'22110' => "Fonz",
			'22111' => "Foradada del Toscar",
			'22112' => "Fraga",
			'22113' => "Fueva, La",
			'22114' => "Gistaín",
			'22115' => "Grado, El",
			'22116' => "Grañén",
			'22117' => "Graus",
			'22119' => "Gurrea de Gállego",
			'22122' => "Hoz de Jaca",
			'22908' => "Hoz y Costean",
			'22124' => "Huerto",
			'22125' => "Huesca",
			'22126' => "Ibieca",
			'22127' => "Igriés",
			'22128' => "Ilche",
			'22129' => "Isábena",
			'22130' => "Jaca",
			'22131' => "Jasa",
			'22133' => "Labuerda",
			'22135' => "Laluenga",
			'22136' => "Lalueza",
			'22137' => "Lanaja",
			'22139' => "Laperdiguera",
			'22141' => "Lascellas-Ponzano",
			'22142' => "Lascuarre",
			'22143' => "Laspaúles",
			'22144' => "Laspuña",
			'22149' => "Loarre",
			'22150' => "Loporzano",
			'22151' => "Loscorrales",
			'22905' => "Lupiñén-Ortilla",
			'22155' => "Monesma y Cajigar",
			'22156' => "Monflorite-Lascasas",
			'22157' => "Montanuy",
			'22158' => "Monzón",
			'22160' => "Naval",
			'22162' => "Novales",
			'22163' => "Nueno",
			'22164' => "Olvena",
			'22165' => "Ontiñena",
			'22167' => "Osso de Cinca",
			'22168' => "Palo",
			'22170' => "Panticosa",
			'22172' => "Peñalba",
			'22173' => "Peñas de Riglos, Las",
			'22174' => "Peralta de Alcofea",
			'22175' => "Peralta de Calasanz",
			'22176' => "Peraltilla",
			'22177' => "Perarrúa",
			'22178' => "Pertusa",
			'22181' => "Piracés",
			'22182' => "Plan",
			'22184' => "Poleñino",
			'22186' => "Pozán de Vero",
			'22187' => "Puebla de Castro, La",
			'22188' => "Puente de Montañana",
			'22902' => "Puente la Reina de Jaca",
			'22189' => "Puértolas",
			'22190' => "Pueyo de Araguás, El",
			'22193' => "Pueyo de Santa Cruz",
			'22195' => "Quicena",
			'22197' => "Robres",
			'22199' => "Sabiñánigo",
			'22200' => "Sahún",
			'22201' => "Salas Altas",
			'22202' => "Salas Bajas",
			'22203' => "Salillas",
			'22204' => "Sallent de Gállego",
			'22205' => "San Esteban de Litera",
			'22207' => "San Juan de Plan",
			'22903' => "San Miguel del Cinca",
			'22206' => "Sangarrén",
			'22208' => "Santa Cilia",
			'22209' => "Santa Cruz de la Serós",
			'22906' => "Santa María de Dulcis",
			'22212' => "Santaliestra y San Quílez",
			'22213' => "Sariñena",
			'22214' => "Secastilla",
			'22215' => "Seira",
			'22217' => "Sena",
			'22218' => "Senés de Alcubierre",
			'22220' => "Sesa",
			'22221' => "Sesué",
			'22222' => "Siétamo",
			'22223' => "Sopeira",
			'22904' => "Sotonera, La",
			'22225' => "Tamarite de Litera",
			'22226' => "Tardienta",
			'22227' => "Tella-Sin",
			'22228' => "Tierz",
			'22229' => "Tolva",
			'22230' => "Torla-Ordesa",
			'22232' => "Torralba de Aragón",
			'22233' => "Torre la Ribera",
			'22234' => "Torrente de Cinca",
			'22235' => "Torres de Alcanadre",
			'22236' => "Torres de Barbués",
			'22239' => "Tramaced",
			'22242' => "Valfarta",
			'22243' => "Valle de Bardají",
			'22901' => "Valle de Hecho",
			'22244' => "Valle de Lierp",
			'22245' => "Velilla de Cinca",
			'22909' => "Vencillón",
			'22247' => "Viacamp y Litera",
			'22248' => "Vicién",
			'22249' => "Villanova",
			'22250' => "Villanúa",
			'22251' => "Villanueva de Sigena",
			'22252' => "Yebra de Basa",
			'22253' => "Yésero",
			'22254' => "Zaidín",
			'23001' => "Albanchez de Mágina",
			'23002' => "Alcalá la Real",
			'23003' => "Alcaudete",
			'23004' => "Aldeaquemada",
			'23005' => "Andújar",
			'23006' => "Arjona",
			'23007' => "Arjonilla",
			'23008' => "Arquillos",
			'23905' => "Arroyo del Ojanco",
			'23009' => "Baeza",
			'23010' => "Bailén",
			'23011' => "Baños de la Encina",
			'23012' => "Beas de Segura",
			'23902' => "Bedmar y Garcíez",
			'23014' => "Begíjar",
			'23015' => "Bélmez de la Moraleda",
			'23016' => "Benatae",
			'23017' => "Cabra del Santo Cristo",
			'23018' => "Cambil",
			'23019' => "Campillo de Arenas",
			'23020' => "Canena",
			'23021' => "Carboneros",
			'23901' => "Cárcheles",
			'23024' => "Carolina, La",
			'23025' => "Castellar",
			'23026' => "Castillo de Locubín",
			'23027' => "Cazalilla",
			'23028' => "Cazorla",
			'23029' => "Chiclana de Segura",
			'23030' => "Chilluévar",
			'23031' => "Escañuela",
			'23032' => "Espeluy",
			'23033' => "Frailes",
			'23034' => "Fuensanta de Martos",
			'23035' => "Fuerte del Rey",
			'23037' => "Génave",
			'23038' => "Guardia de Jaén, La",
			'23039' => "Guarromán",
			'23041' => "Higuera de Calatrava",
			'23042' => "Hinojares",
			'23043' => "Hornos",
			'23044' => "Huelma",
			'23045' => "Huesa",
			'23046' => "Ibros",
			'23047' => "Iruela, La",
			'23048' => "Iznatoraf",
			'23049' => "Jabalquinto",
			'23050' => "Jaén",
			'23051' => "Jamilena",
			'23052' => "Jimena",
			'23053' => "Jódar",
			'23040' => "Lahiguera",
			'23054' => "Larva",
			'23055' => "Linares",
			'23056' => "Lopera",
			'23057' => "Lupión",
			'23058' => "Mancha Real",
			'23059' => "Marmolejo",
			'23060' => "Martos",
			'23061' => "Mengíbar",
			'23062' => "Montizón",
			'23063' => "Navas de San Juan",
			'23064' => "Noalejo",
			'23065' => "Orcera",
			'23066' => "Peal de Becerro",
			'23067' => "Pegalajar",
			'23069' => "Porcuna",
			'23070' => "Pozo Alcón",
			'23071' => "Puente de Génave",
			'23072' => "Puerta de Segura, La",
			'23073' => "Quesada",
			'23074' => "Rus",
			'23075' => "Sabiote",
			'23076' => "Santa Elena",
			'23077' => "Santiago de Calatrava",
			'23904' => "Santiago-Pontones",
			'23079' => "Santisteban del Puerto",
			'23080' => "Santo Tomé",
			'23081' => "Segura de la Sierra",
			'23082' => "Siles",
			'23084' => "Sorihuela del Guadalimar",
			'23085' => "Torreblascopedro",
			'23086' => "Torredelcampo",
			'23087' => "Torredonjimeno",
			'23088' => "Torreperogil",
			'23090' => "Torres",
			'23091' => "Torres de Albánchez",
			'23092' => "Úbeda",
			'23093' => "Valdepeñas de Jaén",
			'23094' => "Vilches",
			'23095' => "Villacarrillo",
			'23096' => "Villanueva de la Reina",
			'23097' => "Villanueva del Arzobispo",
			'23098' => "Villardompardo",
			'23099' => "Villares, Los",
			'23101' => "Villarrodrigo",
			'23903' => "Villatorres",
			'24001' => "Acebedo",
			'24002' => "Algadefe",
			'24003' => "Alija del Infantado",
			'24004' => "Almanza",
			'24005' => "Antigua, La",
			'24006' => "Ardón",
			'24007' => "Arganza",
			'24008' => "Astorga",
			'24009' => "Balboa",
			'24010' => "Bañeza, La",
			'24011' => "Barjas",
			'24012' => "Barrios de Luna, Los",
			'24014' => "Bembibre",
			'24015' => "Benavides",
			'24016' => "Benuza",
			'24017' => "Bercianos del Páramo",
			'24018' => "Bercianos del Real Camino",
			'24019' => "Berlanga del Bierzo",
			'24020' => "Boca de Huérgano",
			'24021' => "Boñar",
			'24022' => "Borrenes",
			'24023' => "Brazuelo",
			'24024' => "Burgo Ranero, El",
			'24025' => "Burón",
			'24026' => "Bustillo del Páramo",
			'24027' => "Cabañas Raras",
			'24028' => "Cabreros del Río",
			'24029' => "Cabrillanes",
			'24030' => "Cacabelos",
			'24031' => "Calzada del Coto",
			'24032' => "Campazas",
			'24033' => "Campo de Villavidel",
			'24034' => "Camponaraya",
			'24037' => "Cármenes",
			'24038' => "Carracedelo",
			'24039' => "Carrizo",
			'24040' => "Carrocera",
			'24041' => "Carucedo",
			'24042' => "Castilfalé",
			'24043' => "Castrillo de Cabrera",
			'24044' => "Castrillo de la Valduerna",
			'24046' => "Castrocalbón",
			'24047' => "Castrocontrigo",
			'24049' => "Castropodame",
			'24050' => "Castrotierra de Valmadrigal",
			'24051' => "Cea",
			'24052' => "Cebanico",
			'24053' => "Cebrones del Río",
			'24065' => "Chozas de Abajo",
			'24054' => "Cimanes de la Vega",
			'24055' => "Cimanes del Tejar",
			'24056' => "Cistierna",
			'24057' => "Congosto",
			'24058' => "Corbillos de los Oteros",
			'24059' => "Corullón",
			'24060' => "Crémenes",
			'24061' => "Cuadros",
			'24062' => "Cubillas de los Oteros",
			'24063' => "Cubillas de Rueda",
			'24064' => "Cubillos del Sil",
			'24066' => "Destriana",
			'24067' => "Encinedo",
			'24068' => "Ercina, La",
			'24069' => "Escobar de Campos",
			'24070' => "Fabero",
			'24071' => "Folgoso de la Ribera",
			'24073' => "Fresno de la Vega",
			'24074' => "Fuentes de Carbajal",
			'24076' => "Garrafe de Torío",
			'24077' => "Gordaliza del Pino",
			'24078' => "Gordoncillo",
			'24079' => "Gradefes",
			'24080' => "Grajal de Campos",
			'24081' => "Gusendos de los Oteros",
			'24082' => "Hospital de Órbigo",
			'24083' => "Igüeña",
			'24084' => "Izagre",
			'24086' => "Joarilla de las Matas",
			'24087' => "Laguna Dalga",
			'24088' => "Laguna de Negrillos",
			'24089' => "León",
			'24092' => "Llamas de la Ribera",
			'24090' => "Lucillo",
			'24091' => "Luyego",
			'24093' => "Magaz de Cepeda",
			'24094' => "Mansilla de las Mulas",
			'24095' => "Mansilla Mayor",
			'24096' => "Maraña",
			'24097' => "Matadeón de los Oteros",
			'24098' => "Matallana de Torío",
			'24099' => "Matanza",
			'24100' => "Molinaseca",
			'24101' => "Murias de Paredes",
			'24102' => "Noceda del Bierzo",
			'24103' => "Oencia",
			'24104' => "Omañas, Las",
			'24105' => "Onzonilla",
			'24106' => "Oseja de Sajambre",
			'24107' => "Pajares de los Oteros",
			'24108' => "Palacios de la Valduerna",
			'24109' => "Palacios del Sil",
			'24110' => "Páramo del Sil",
			'24112' => "Peranzanes",
			'24113' => "Pobladura de Pelayo García",
			'24114' => "Pola de Gordón, La",
			'24115' => "Ponferrada",
			'24116' => "Posada de Valdeón",
			'24117' => "Pozuelo del Páramo",
			'24118' => "Prado de la Guzpeña",
			'24119' => "Priaranza del Bierzo",
			'24120' => "Prioro",
			'24121' => "Puebla de Lillo",
			'24122' => "Puente de Domingo Flórez",
			'24123' => "Quintana del Castillo",
			'24124' => "Quintana del Marco",
			'24125' => "Quintana y Congosto",
			'24127' => "Regueras de Arriba",
			'24129' => "Reyero",
			'24130' => "Riaño",
			'24131' => "Riego de la Vega",
			'24132' => "Riello",
			'24133' => "Rioseco de Tapia",
			'24134' => "Robla, La",
			'24136' => "Roperuelos del Páramo",
			'24137' => "Sabero",
			'24139' => "Sahagún",
			'24141' => "San Adrián del Valle",
			'24142' => "San Andrés del Rabanedo",
			'24144' => "San Cristóbal de la Polantera",
			'24145' => "San Emiliano",
			'24146' => "San Esteban de Nogales",
			'24148' => "San Justo de la Vega",
			'24149' => "San Millán de los Caballeros",
			'24150' => "San Pedro Bercianos",
			'24143' => "Sancedo",
			'24151' => "Santa Colomba de Curueño",
			'24152' => "Santa Colomba de Somoza",
			'24153' => "Santa Cristina de Valmadrigal",
			'24154' => "Santa Elena de Jamuz",
			'24155' => "Santa María de la Isla",
			'24158' => "Santa María de Ordás",
			'24156' => "Santa María del Monte de Cea",
			'24157' => "Santa María del Páramo",
			'24159' => "Santa Marina del Rey",
			'24160' => "Santas Martas",
			'24161' => "Santiago Millas",
			'24162' => "Santovenia de la Valdoncina",
			'24163' => "Sariegos",
			'24164' => "Sena de Luna",
			'24165' => "Sobrado",
			'24166' => "Soto de la Vega",
			'24167' => "Soto y Amío",
			'24168' => "Toral de los Guzmanes",
			'24206' => "Toral de los Vados",
			'24169' => "Toreno",
			'24170' => "Torre del Bierzo",
			'24171' => "Trabadelo",
			'24172' => "Truchas",
			'24173' => "Turcia",
			'24174' => "Urdiales del Páramo",
			'24185' => "Val de San Lorenzo",
			'24175' => "Valdefresno",
			'24176' => "Valdefuentes del Páramo",
			'24177' => "Valdelugueros",
			'24178' => "Valdemora",
			'24179' => "Valdepiélago",
			'24180' => "Valdepolo",
			'24181' => "Valderas",
			'24182' => "Valderrey",
			'24183' => "Valderrueda",
			'24184' => "Valdesamario",
			'24187' => "Valdevimbre",
			'24188' => "Valencia de Don Juan",
			'24036' => "Valle de Ancares",
			'24191' => "Vallecillo",
			'24189' => "Valverde de la Virgen",
			'24190' => "Valverde-Enrique",
			'24193' => "Vecilla, La",
			'24196' => "Vega de Espinareda",
			'24197' => "Vega de Infanzones",
			'24198' => "Vega de Valcarce",
			'24194' => "Vegacervera",
			'24199' => "Vegaquemada",
			'24201' => "Vegas del Condado",
			'24202' => "Villablino",
			'24203' => "Villabraz",
			'24205' => "Villadangos del Páramo",
			'24207' => "Villademor de la Vega",
			'24209' => "Villafranca del Bierzo",
			'24210' => "Villagatón",
			'24211' => "Villamandos",
			'24901' => "Villamanín",
			'24212' => "Villamañán",
			'24213' => "Villamartín de Don Sancho",
			'24214' => "Villamejil",
			'24215' => "Villamol",
			'24216' => "Villamontán de la Valduerna",
			'24217' => "Villamoratiel de las Matas",
			'24218' => "Villanueva de las Manzanas",
			'24219' => "Villaobispo de Otero",
			'24902' => "Villaornate y Castro",
			'24221' => "Villaquejida",
			'24222' => "Villaquilambre",
			'24223' => "Villarejo de Órbigo",
			'24224' => "Villares de Órbigo",
			'24225' => "Villasabariego",
			'24226' => "Villaselán",
			'24227' => "Villaturiel",
			'24228' => "Villazala",
			'24229' => "Villazanzo de Valderaduey",
			'24230' => "Zotes del Páramo",
			'25001' => "Abella de la Conca",
			'25002' => "Àger",
			'25003' => "Agramunt",
			'25038' => "Aitona",
			'25004' => "Alamús, Els",
			'25005' => "Alàs i Cerc",
			'25006' => "Albagés, L'",
			'25007' => "Albatàrrec",
			'25008' => "Albesa",
			'25009' => "Albi, L'",
			'25010' => "Alcanó",
			'25011' => "Alcarràs",
			'25012' => "Alcoletge",
			'25013' => "Alfarràs",
			'25014' => "Alfés",
			'25015' => "Algerri",
			'25016' => "Alguaire",
			'25017' => "Alins",
			'25019' => "Almacelles",
			'25020' => "Almatret",
			'25021' => "Almenar",
			'25022' => "Alòs de Balaguer",
			'25023' => "Alpicat",
			'25024' => "Alt Àneu",
			'25027' => "Anglesola",
			'25029' => "Arbeca",
			'25031' => "Arres",
			'25032' => "Arsèguel",
			'25033' => "Artesa de Lleida",
			'25034' => "Artesa de Segre",
			'25036' => "Aspa",
			'25037' => "Avellanes i Santa Linya, Les",
			'25039' => "Baix Pallars",
			'25040' => "Balaguer",
			'25041' => "Barbens",
			'25042' => "Baronia de Rialb, La",
			'25044' => "Bassella",
			'25045' => "Bausen",
			'25046' => "Belianes",
			'25170' => "Bellaguarda",
			'25047' => "Bellcaire d'Urgell",
			'25048' => "Bell-lloc d'Urgell",
			'25049' => "Bellmunt d'Urgell",
			'25050' => "Bellpuig",
			'25051' => "Bellver de Cerdanya",
			'25052' => "Bellvís",
			'25053' => "Benavent de Segrià",
			'25055' => "Biosca",
			'25057' => "Bòrdes, Es",
			'25058' => "Borges Blanques, Les",
			'25059' => "Bossòst",
			'25056' => "Bovera",
			'25060' => "Cabanabona",
			'25061' => "Cabó",
			'25062' => "Camarasa",
			'25063' => "Canejan",
			'25904' => "Castell de Mur",
			'25064' => "Castellar de la Ribera",
			'25067' => "Castelldans",
			'25068' => "Castellnou de Seana",
			'25069' => "Castelló de Farfanya",
			'25070' => "Castellserà",
			'25071' => "Cava",
			'25072' => "Cervera",
			'25073' => "Cervià de les Garrigues",
			'25074' => "Ciutadilla",
			'25075' => "Clariana de Cardener",
			'25076' => "Cogul, El",
			'25077' => "Coll de Nargó",
			'25163' => "Coma i la Pedra, La",
			'25161' => "Conca de Dalt",
			'25078' => "Corbins",
			'25079' => "Cubells",
			'25081' => "Espluga Calba, L'",
			'25082' => "Espot",
			'25088' => "Estamariu",
			'25085' => "Estaràs",
			'25086' => "Esterri d'Àneu",
			'25087' => "Esterri de Cardós",
			'25089' => "Farrera",
			'25908' => "Fígols i Alinyà",
			'25092' => "Floresta, La",
			'25093' => "Fondarella",
			'25094' => "Foradada",
			'25096' => "Fuliola, La",
			'25097' => "Fulleda",
			'25098' => "Gavet de la Conca",
			'25912' => "Gimenells i el Pla de la Font",
			'25099' => "Golmés",
			'25100' => "Gósol",
			'25101' => "Granadella, La",
			'25102' => "Granja d'Escarp, La",
			'25103' => "Granyanella",
			'25105' => "Granyena de les Garrigues",
			'25104' => "Granyena de Segarra",
			'25109' => "Guimerà",
			'25903' => "Guingueta d'Àneu, La",
			'25110' => "Guissona",
			'25111' => "Guixers",
			'25115' => "Isona i Conca Dellà",
			'25112' => "Ivars de Noguera",
			'25113' => "Ivars d'Urgell",
			'25114' => "Ivorra",
			'25910' => "Josa i Tuixén",
			'25118' => "Juncosa",
			'25119' => "Juneda",
			'25121' => "Les",
			'25122' => "Linyola",
			'25123' => "Lladorre",
			'25124' => "Lladurs",
			'25125' => "Llardecans",
			'25126' => "Llavorsí",
			'25120' => "Lleida",
			'25127' => "Lles de Cerdanya",
			'25128' => "Llimiana",
			'25129' => "Llobera",
			'25133' => "Maials",
			'25130' => "Maldà",
			'25131' => "Massalcoreig",
			'25132' => "Massoteres",
			'25134' => "Menàrguens",
			'25135' => "Miralcamp",
			'25137' => "Mollerussa",
			'25136' => "Molsosa, La",
			'25139' => "Montellà i Martinet",
			'25140' => "Montferrer i Castellbò",
			'25138' => "Montgai",
			'25142' => "Montoliu de Lleida",
			'25141' => "Montoliu de Segarra",
			'25143' => "Montornès de Segarra",
			'25145' => "Nalec",
			'25025' => "Naut Aran",
			'25146' => "Navès",
			'25148' => "Odèn",
			'25149' => "Oliana",
			'25150' => "Oliola",
			'25151' => "Olius",
			'25152' => "Oluges, Les",
			'25153' => "Omellons, Els",
			'25154' => "Omells de na Gaia, Els",
			'25155' => "Organyà",
			'25156' => "Os de Balaguer",
			'25157' => "Ossó de Sió",
			'25158' => "Palau d'Anglesola, El",
			'25164' => "Penelles",
			'25165' => "Peramola",
			'25166' => "Pinell de Solsonès",
			'25167' => "Pinós",
			'25911' => "Plans de Sió, Els",
			'25168' => "Poal, El",
			'25169' => "Pobla de Cérvoles, La",
			'25171' => "Pobla de Segur, La",
			'25030' => "Pont de Bar, El",
			'25173' => "Pont de Suert, El",
			'25172' => "Ponts",
			'25174' => "Portella, La",
			'25175' => "Prats i Sansor",
			'25176' => "Preixana",
			'25177' => "Preixens",
			'25179' => "Prullans",
			'25180' => "Puiggròs",
			'25181' => "Puigverd d'Agramunt",
			'25182' => "Puigverd de Lleida",
			'25183' => "Rialp",
			'25905' => "Ribera d'Ondara",
			'25185' => "Ribera d'Urgellet",
			'25186' => "Riner",
			'25913' => "Riu de Cerdanya",
			'25189' => "Rosselló",
			'25190' => "Salàs de Pallars",
			'25191' => "Sanaüja",
			'25196' => "Sant Esteve de la Sarga",
			'25192' => "Sant Guim de Freixenet",
			'25197' => "Sant Guim de la Plana",
			'25193' => "Sant Llorenç de Morunys",
			'25902' => "Sant Martí de Riucorb",
			'25194' => "Sant Ramon",
			'25201' => "Sarroca de Bellera",
			'25200' => "Sarroca de Lleida",
			'25202' => "Senterada",
			'25035' => "Sentiu de Sió, La",
			'25204' => "Seròs",
			'25203' => "Seu d'Urgell, La",
			'25205' => "Sidamon",
			'25206' => "Soleràs, El",
			'25207' => "Solsona",
			'25208' => "Soriguera",
			'25209' => "Sort",
			'25210' => "Soses",
			'25211' => "Sudanell",
			'25212' => "Sunyer",
			'25215' => "Talarn",
			'25216' => "Talavera",
			'25217' => "Tàrrega",
			'25218' => "Tarrés",
			'25219' => "Tarroja de Segarra",
			'25220' => "Térmens",
			'25221' => "Tírvia",
			'25222' => "Tiurana",
			'25223' => "Torà",
			'25224' => "Torms, Els",
			'25225' => "Tornabous",
			'25227' => "Torre de Cabdella, La",
			'25226' => "Torrebesses",
			'25228' => "Torrefarrera",
			'25907' => "Torrefeta i Florejacs",
			'25230' => "Torregrossa",
			'25231' => "Torrelameu",
			'25232' => "Torres de Segre",
			'25233' => "Torre-serona",
			'25234' => "Tremp",
			'25043' => "Vall de Boí, La",
			'25901' => "Vall de Cardós",
			'25238' => "Vallbona de les Monges",
			'25240' => "Vallfogona de Balaguer",
			'25906' => "Valls d'Aguilar, Les",
			'25239' => "Valls de Valira, Les",
			'25909' => "Vansa i Fórnols, La",
			'25242' => "Verdú",
			'25243' => "Vielha e Mijaran",
			'25244' => "Vilagrassa",
			'25245' => "Vilaller",
			'25247' => "Vilamòs",
			'25248' => "Vilanova de Bellpuig",
			'25254' => "Vilanova de la Barca",
			'25249' => "Vilanova de l'Aguda",
			'25250' => "Vilanova de Meià",
			'25251' => "Vilanova de Segrià",
			'25252' => "Vila-sana",
			'25253' => "Vilosell, El",
			'25255' => "Vinaixa",
			'26001' => "Ábalos",
			'26002' => "Agoncillo",
			'26003' => "Aguilar del Río Alhama",
			'26004' => "Ajamil de Cameros",
			'26005' => "Albelda de Iregua",
			'26006' => "Alberite",
			'26007' => "Alcanadre",
			'26008' => "Aldeanueva de Ebro",
			'26009' => "Alesanco",
			'26010' => "Alesón",
			'26011' => "Alfaro",
			'26012' => "Almarza de Cameros",
			'26013' => "Anguciana",
			'26014' => "Anguiano",
			'26015' => "Arenzana de Abajo",
			'26016' => "Arenzana de Arriba",
			'26017' => "Arnedillo",
			'26018' => "Arnedo",
			'26019' => "Arrúbal",
			'26020' => "Ausejo",
			'26021' => "Autol",
			'26022' => "Azofra",
			'26023' => "Badarán",
			'26024' => "Bañares",
			'26026' => "Baños de Río Tobía",
			'26025' => "Baños de Rioja",
			'26027' => "Berceo",
			'26028' => "Bergasa",
			'26029' => "Bergasillas Bajera",
			'26030' => "Bezares",
			'26031' => "Bobadilla",
			'26032' => "Brieva de Cameros",
			'26033' => "Briñas",
			'26034' => "Briones",
			'26035' => "Cabezón de Cameros",
			'26036' => "Calahorra",
			'26037' => "Camprovín",
			'26038' => "Canales de la Sierra",
			'26039' => "Canillas de Río Tuerto",
			'26040' => "Cañas",
			'26041' => "Cárdenas",
			'26042' => "Casalarreina",
			'26043' => "Castañares de Rioja",
			'26044' => "Castroviejo",
			'26045' => "Cellorigo",
			'26046' => "Cenicero",
			'26047' => "Cervera del Río Alhama",
			'26048' => "Cidamón",
			'26049' => "Cihuri",
			'26050' => "Cirueña",
			'26051' => "Clavijo",
			'26052' => "Cordovín",
			'26053' => "Corera",
			'26054' => "Cornago",
			'26055' => "Corporales",
			'26056' => "Cuzcurrita de Río Tirón",
			'26057' => "Daroca de Rioja",
			'26058' => "Enciso",
			'26059' => "Entrena",
			'26060' => "Estollo",
			'26061' => "Ezcaray",
			'26062' => "Foncea",
			'26063' => "Fonzaleche",
			'26064' => "Fuenmayor",
			'26065' => "Galbárruli",
			'26066' => "Galilea",
			'26067' => "Gallinero de Cameros",
			'26068' => "Gimileo",
			'26069' => "Grañón",
			'26070' => "Grávalos",
			'26071' => "Haro",
			'26072' => "Herce",
			'26073' => "Herramélluri",
			'26074' => "Hervías",
			'26075' => "Hormilla",
			'26076' => "Hormilleja",
			'26077' => "Hornillos de Cameros",
			'26078' => "Hornos de Moncalvillo",
			'26079' => "Huércanos",
			'26080' => "Igea",
			'26081' => "Jalón de Cameros",
			'26082' => "Laguna de Cameros",
			'26083' => "Lagunilla del Jubera",
			'26084' => "Lardero",
			'26086' => "Ledesma de la Cogolla",
			'26087' => "Leiva",
			'26088' => "Leza de Río Leza",
			'26089' => "Logroño",
			'26091' => "Lumbreras de Cameros",
			'26092' => "Manjarrés",
			'26093' => "Mansilla de la Sierra",
			'26094' => "Manzanares de Rioja",
			'26095' => "Matute",
			'26096' => "Medrano",
			'26098' => "Munilla",
			'26099' => "Murillo de Río Leza",
			'26100' => "Muro de Aguas",
			'26101' => "Muro en Cameros",
			'26102' => "Nájera",
			'26103' => "Nalda",
			'26104' => "Navajún",
			'26105' => "Navarrete",
			'26106' => "Nestares",
			'26107' => "Nieva de Cameros",
			'26109' => "Ochánduri",
			'26108' => "Ocón",
			'26110' => "Ojacastro",
			'26111' => "Ollauri",
			'26112' => "Ortigosa de Cameros",
			'26113' => "Pazuengos",
			'26114' => "Pedroso",
			'26115' => "Pinillos",
			'26117' => "Pradejón",
			'26118' => "Pradillo",
			'26119' => "Préjano",
			'26120' => "Quel",
			'26121' => "Rabanera",
			'26122' => "Rasillo de Cameros, El",
			'26123' => "Redal, El",
			'26124' => "Ribafrecha",
			'26125' => "Rincón de Soto",
			'26126' => "Robres del Castillo",
			'26127' => "Rodezno",
			'26128' => "Sajazarra",
			'26129' => "San Asensio",
			'26130' => "San Millán de la Cogolla",
			'26131' => "San Millán de Yécora",
			'26132' => "San Román de Cameros",
			'26139' => "San Torcuato",
			'26142' => "San Vicente de la Sonsierra",
			'26134' => "Santa Coloma",
			'26135' => "Santa Engracia del Jubera",
			'26136' => "Santa Eulalia Bajera",
			'26138' => "Santo Domingo de la Calzada",
			'26140' => "Santurde de Rioja",
			'26141' => "Santurdejo",
			'26143' => "Sojuela",
			'26144' => "Sorzano",
			'26145' => "Sotés",
			'26146' => "Soto en Cameros",
			'26147' => "Terroba",
			'26148' => "Tirgo",
			'26149' => "Tobía",
			'26150' => "Tormantos",
			'26153' => "Torre en Cameros",
			'26151' => "Torrecilla en Cameros",
			'26152' => "Torrecilla sobre Alesanco",
			'26154' => "Torremontalbo",
			'26155' => "Treviana",
			'26157' => "Tricio",
			'26158' => "Tudelilla",
			'26160' => "Uruñuela",
			'26161' => "Valdemadera",
			'26162' => "Valgañón",
			'26163' => "Ventosa",
			'26164' => "Ventrosa",
			'26165' => "Viguera",
			'26166' => "Villalba de Rioja",
			'26167' => "Villalobar de Rioja",
			'26168' => "Villamediana de Iregua",
			'26169' => "Villanueva de Cameros",
			'26170' => "Villar de Arnedo, El",
			'26171' => "Villar de Torre",
			'26172' => "Villarejo",
			'26173' => "Villarroya",
			'26174' => "Villarta-Quintana",
			'26175' => "Villavelayo",
			'26176' => "Villaverde de Rioja",
			'26177' => "Villoslada de Cameros",
			'26178' => "Viniegra de Abajo",
			'26179' => "Viniegra de Arriba",
			'26180' => "Zarratón",
			'26181' => "Zarzosa",
			'26183' => "Zorraquín",
			'27001' => "Abadín",
			'27002' => "Alfoz",
			'27003' => "Antas de Ulla",
			'27004' => "Baleira",
			'27901' => "Baralla",
			'27005' => "Barreiros",
			'27006' => "Becerreá",
			'27007' => "Begonte",
			'27008' => "Bóveda",
			'27902' => "Burela",
			'27009' => "Carballedo",
			'27010' => "Castro de Rei",
			'27011' => "Castroverde",
			'27012' => "Cervantes",
			'27013' => "Cervo",
			'27016' => "Chantada",
			'27014' => "Corgo, O",
			'27015' => "Cospeito",
			'27017' => "Folgoso do Courel",
			'27018' => "Fonsagrada, A",
			'27019' => "Foz",
			'27020' => "Friol",
			'27022' => "Guitiriz",
			'27023' => "Guntín",
			'27024' => "Incio, O",
			'27026' => "Láncara",
			'27027' => "Lourenzá",
			'27028' => "Lugo",
			'27029' => "Meira",
			'27030' => "Mondoñedo",
			'27031' => "Monforte de Lemos",
			'27032' => "Monterroso",
			'27033' => "Muras",
			'27034' => "Navia de Suarna",
			'27035' => "Negueira de Muñiz",
			'27037' => "Nogais, As",
			'27038' => "Ourol",
			'27039' => "Outeiro de Rei",
			'27040' => "Palas de Rei",
			'27041' => "Pantón",
			'27042' => "Paradela",
			'27043' => "Páramo, O",
			'27044' => "Pastoriza, A",
			'27045' => "Pedrafita do Cebreiro",
			'27047' => "Pobra do Brollón, A",
			'27046' => "Pol",
			'27048' => "Pontenova, A",
			'27049' => "Portomarín",
			'27050' => "Quiroga",
			'27056' => "Rábade",
			'27051' => "Ribadeo",
			'27052' => "Ribas de Sil",
			'27053' => "Ribeira de Piquín",
			'27054' => "Riotorto",
			'27055' => "Samos",
			'27057' => "Sarria",
			'27058' => "Saviñao, O",
			'27059' => "Sober",
			'27060' => "Taboada",
			'27061' => "Trabada",
			'27062' => "Triacastela",
			'27063' => "Valadouro, O",
			'27064' => "Vicedo, O",
			'27065' => "Vilalba",
			'27066' => "Viveiro",
			'27021' => "Xermade",
			'27025' => "Xove",
			'28001' => "Acebeda, La",
			'28002' => "Ajalvir",
			'28003' => "Alameda del Valle",
			'28004' => "Álamo, El",
			'28005' => "Alcalá de Henares",
			'28006' => "Alcobendas",
			'28007' => "Alcorcón",
			'28008' => "Aldea del Fresno",
			'28009' => "Algete",
			'28010' => "Alpedrete",
			'28011' => "Ambite",
			'28012' => "Anchuelo",
			'28013' => "Aranjuez",
			'28014' => "Arganda del Rey",
			'28015' => "Arroyomolinos",
			'28016' => "Atazar, El",
			'28017' => "Batres",
			'28018' => "Becerril de la Sierra",
			'28019' => "Belmonte de Tajo",
			'28021' => "Berrueco, El",
			'28020' => "Berzosa del Lozoya",
			'28022' => "Boadilla del Monte",
			'28023' => "Boalo, El",
			'28024' => "Braojos",
			'28025' => "Brea de Tajo",
			'28026' => "Brunete",
			'28027' => "Buitrago del Lozoya",
			'28028' => "Bustarviejo",
			'28029' => "Cabanillas de la Sierra",
			'28030' => "Cabrera, La",
			'28031' => "Cadalso de los Vidrios",
			'28032' => "Camarma de Esteruelas",
			'28033' => "Campo Real",
			'28034' => "Canencia",
			'28035' => "Carabaña",
			'28036' => "Casarrubuelos",
			'28037' => "Cenicientos",
			'28038' => "Cercedilla",
			'28039' => "Cervera de Buitrago",
			'28051' => "Chapinería",
			'28052' => "Chinchón",
			'28040' => "Ciempozuelos",
			'28041' => "Cobeña",
			'28046' => "Collado Mediano",
			'28047' => "Collado Villalba",
			'28043' => "Colmenar de Oreja",
			'28042' => "Colmenar del Arroyo",
			'28045' => "Colmenar Viejo",
			'28044' => "Colmenarejo",
			'28048' => "Corpa",
			'28049' => "Coslada",
			'28050' => "Cubas de la Sagra",
			'28053' => "Daganzo de Arriba",
			'28054' => "Escorial, El",
			'28055' => "Estremera",
			'28056' => "Fresnedillas de la Oliva",
			'28057' => "Fresno de Torote",
			'28058' => "Fuenlabrada",
			'28059' => "Fuente el Saz de Jarama",
			'28060' => "Fuentidueña de Tajo",
			'28061' => "Galapagar",
			'28062' => "Garganta de los Montes",
			'28063' => "Gargantilla del Lozoya y Pinilla de Buitrago",
			'28064' => "Gascones",
			'28065' => "Getafe",
			'28066' => "Griñón",
			'28067' => "Guadalix de la Sierra",
			'28068' => "Guadarrama",
			'28069' => "Hiruela, La",
			'28070' => "Horcajo de la Sierra-Aoslos",
			'28071' => "Horcajuelo de la Sierra",
			'28072' => "Hoyo de Manzanares",
			'28073' => "Humanes de Madrid",
			'28074' => "Leganés",
			'28075' => "Loeches",
			'28076' => "Lozoya",
			'28901' => "Lozoyuela-Navas-Sieteiglesias",
			'28078' => "Madarcos",
			'28079' => "Madrid",
			'28080' => "Majadahonda",
			'28082' => "Manzanares el Real",
			'28083' => "Meco",
			'28084' => "Mejorada del Campo",
			'28085' => "Miraflores de la Sierra",
			'28086' => "Molar, El",
			'28087' => "Molinos, Los",
			'28088' => "Montejo de la Sierra",
			'28089' => "Moraleja de Enmedio",
			'28090' => "Moralzarzal",
			'28091' => "Morata de Tajuña",
			'28092' => "Móstoles",
			'28093' => "Navacerrada",
			'28094' => "Navalafuente",
			'28095' => "Navalagamella",
			'28096' => "Navalcarnero",
			'28097' => "Navarredonda y San Mamés",
			'28099' => "Navas del Rey",
			'28100' => "Nuevo Baztán",
			'28101' => "Olmeda de las Fuentes",
			'28102' => "Orusco de Tajuña",
			'28104' => "Paracuellos de Jarama",
			'28106' => "Parla",
			'28107' => "Patones",
			'28108' => "Pedrezuela",
			'28109' => "Pelayos de la Presa",
			'28110' => "Perales de Tajuña",
			'28111' => "Pezuela de las Torres",
			'28112' => "Pinilla del Valle",
			'28113' => "Pinto",
			'28114' => "Piñuécar-Gandullas",
			'28115' => "Pozuelo de Alarcón",
			'28116' => "Pozuelo del Rey",
			'28117' => "Prádena del Rincón",
			'28118' => "Puebla de la Sierra",
			'28902' => "Puentes Viejas",
			'28119' => "Quijorna",
			'28120' => "Rascafría",
			'28121' => "Redueña",
			'28122' => "Ribatejada",
			'28123' => "Rivas-Vaciamadrid",
			'28124' => "Robledillo de la Jara",
			'28125' => "Robledo de Chavela",
			'28126' => "Robregordo",
			'28127' => "Rozas de Madrid, Las",
			'28128' => "Rozas de Puerto Real",
			'28129' => "San Agustín del Guadalix",
			'28130' => "San Fernando de Henares",
			'28131' => "San Lorenzo de El Escorial",
			'28132' => "San Martín de la Vega",
			'28133' => "San Martín de Valdeiglesias",
			'28134' => "San Sebastián de los Reyes",
			'28135' => "Santa María de la Alameda",
			'28136' => "Santorcaz",
			'28137' => "Santos de la Humosa, Los",
			'28138' => "Serna del Monte, La",
			'28140' => "Serranillos del Valle",
			'28141' => "Sevilla la Nueva",
			'28143' => "Somosierra",
			'28144' => "Soto del Real",
			'28145' => "Talamanca de Jarama",
			'28146' => "Tielmes",
			'28147' => "Titulcia",
			'28148' => "Torrejón de Ardoz",
			'28149' => "Torrejón de la Calzada",
			'28150' => "Torrejón de Velasco",
			'28151' => "Torrelaguna",
			'28152' => "Torrelodones",
			'28153' => "Torremocha de Jarama",
			'28154' => "Torres de la Alameda",
			'28903' => "Tres Cantos",
			'28155' => "Valdaracete",
			'28156' => "Valdeavero",
			'28157' => "Valdelaguna",
			'28158' => "Valdemanco",
			'28159' => "Valdemaqueda",
			'28160' => "Valdemorillo",
			'28161' => "Valdemoro",
			'28162' => "Valdeolmos-Alalpardo",
			'28163' => "Valdepiélagos",
			'28164' => "Valdetorres de Jarama",
			'28165' => "Valdilecha",
			'28166' => "Valverde de Alcalá",
			'28167' => "Velilla de San Antonio",
			'28168' => "Vellón, El",
			'28169' => "Venturada",
			'28171' => "Villa del Prado",
			'28170' => "Villaconejos",
			'28172' => "Villalbilla",
			'28173' => "Villamanrique de Tajo",
			'28174' => "Villamanta",
			'28175' => "Villamantilla",
			'28176' => "Villanueva de la Cañada",
			'28178' => "Villanueva de Perales",
			'28177' => "Villanueva del Pardillo",
			'28179' => "Villar del Olmo",
			'28180' => "Villarejo de Salvanés",
			'28181' => "Villaviciosa de Odón",
			'28182' => "Villavieja del Lozoya",
			'28183' => "Zarzalejo",
			'29001' => "Alameda",
			'29002' => "Alcaucín",
			'29003' => "Alfarnate",
			'29004' => "Alfarnatejo",
			'29005' => "Algarrobo",
			'29006' => "Algatocín",
			'29007' => "Alhaurín de la Torre",
			'29008' => "Alhaurín el Grande",
			'29009' => "Almáchar",
			'29010' => "Almargen",
			'29011' => "Almogía",
			'29012' => "Álora",
			'29013' => "Alozaina",
			'29014' => "Alpandeire",
			'29015' => "Antequera",
			'29016' => "Árchez",
			'29017' => "Archidona",
			'29018' => "Ardales",
			'29019' => "Arenas",
			'29020' => "Arriate",
			'29021' => "Atajate",
			'29022' => "Benadalid",
			'29023' => "Benahavís",
			'29024' => "Benalauría",
			'29025' => "Benalmádena",
			'29026' => "Benamargosa",
			'29027' => "Benamocarra",
			'29028' => "Benaoján",
			'29029' => "Benarrabá",
			'29030' => "Borge, El",
			'29031' => "Burgo, El",
			'29032' => "Campillos",
			'29033' => "Canillas de Aceituno",
			'29034' => "Canillas de Albaida",
			'29035' => "Cañete la Real",
			'29036' => "Carratraca",
			'29037' => "Cartajima",
			'29038' => "Cártama",
			'29039' => "Casabermeja",
			'29040' => "Casarabonela",
			'29041' => "Casares",
			'29042' => "Coín",
			'29043' => "Colmenar",
			'29044' => "Comares",
			'29045' => "Cómpeta",
			'29046' => "Cortes de la Frontera",
			'29047' => "Cuevas Bajas",
			'29049' => "Cuevas de San Marcos",
			'29048' => "Cuevas del Becerro",
			'29050' => "Cútar",
			'29051' => "Estepona",
			'29052' => "Faraján",
			'29053' => "Frigiliana",
			'29054' => "Fuengirola",
			'29055' => "Fuente de Piedra",
			'29056' => "Gaucín",
			'29057' => "Genalguacil",
			'29058' => "Guaro",
			'29059' => "Humilladero",
			'29060' => "Igualeja",
			'29061' => "Istán",
			'29062' => "Iznate",
			'29063' => "Jimera de Líbar",
			'29064' => "Jubrique",
			'29065' => "Júzcar",
			'29066' => "Macharaviaya",
			'29067' => "Málaga",
			'29068' => "Manilva",
			'29069' => "Marbella",
			'29070' => "Mijas",
			'29071' => "Moclinejo",
			'29072' => "Mollina",
			'29073' => "Monda",
			'29903' => "Montecorto",
			'29074' => "Montejaque",
			'29075' => "Nerja",
			'29076' => "Ojén",
			'29077' => "Parauta",
			'29079' => "Periana",
			'29080' => "Pizarra",
			'29081' => "Pujerra",
			'29082' => "Rincón de la Victoria",
			'29083' => "Riogordo",
			'29084' => "Ronda",
			'29085' => "Salares",
			'29086' => "Sayalonga",
			'29087' => "Sedella",
			'29904' => "Serrato",
			'29088' => "Sierra de Yeguas",
			'29089' => "Teba",
			'29090' => "Tolox",
			'29901' => "Torremolinos",
			'29091' => "Torrox",
			'29092' => "Totalán",
			'29093' => "Valle de Abdalajís",
			'29094' => "Vélez-Málaga",
			'29095' => "Villanueva de Algaidas",
			'29902' => "Villanueva de la Concepción",
			'29098' => "Villanueva de Tapia",
			'29096' => "Villanueva del Rosario",
			'29097' => "Villanueva del Trabuco",
			'29099' => "Viñuela",
			'29100' => "Yunquera",
			'30001' => "Abanilla",
			'30002' => "Abarán",
			'30003' => "Águilas",
			'30004' => "Albudeite",
			'30005' => "Alcantarilla",
			'30902' => "Alcázares, Los",
			'30006' => "Aledo",
			'30007' => "Alguazas",
			'30008' => "Alhama de Murcia",
			'30009' => "Archena",
			'30010' => "Beniel",
			'30011' => "Blanca",
			'30012' => "Bullas",
			'30013' => "Calasparra",
			'30014' => "Campos del Río",
			'30015' => "Caravaca de la Cruz",
			'30016' => "Cartagena",
			'30017' => "Cehegín",
			'30018' => "Ceutí",
			'30019' => "Cieza",
			'30020' => "Fortuna",
			'30021' => "Fuente Álamo de Murcia",
			'30022' => "Jumilla",
			'30023' => "Librilla",
			'30024' => "Lorca",
			'30025' => "Lorquí",
			'30026' => "Mazarrón",
			'30027' => "Molina de Segura",
			'30028' => "Moratalla",
			'30029' => "Mula",
			'30030' => "Murcia",
			'30031' => "Ojós",
			'30032' => "Pliego",
			'30033' => "Puerto Lumbreras",
			'30034' => "Ricote",
			'30035' => "San Javier",
			'30036' => "San Pedro del Pinatar",
			'30901' => "Santomera",
			'30037' => "Torre-Pacheco",
			'30038' => "Torres de Cotillas, Las",
			'30039' => "Totana",
			'30040' => "Ulea",
			'30041' => "Unión, La",
			'30042' => "Villanueva del Río Segura",
			'30043' => "Yecla",
			'31001' => "Abáigar",
			'31002' => "Abárzuza/Abartzuza",
			'31003' => "Abaurregaina/Abaurrea Alta",
			'31004' => "Abaurrepea/Abaurrea Baja",
			'31005' => "Aberin",
			'31006' => "Ablitas",
			'31007' => "Adiós",
			'31008' => "Aguilar de Codés",
			'31009' => "Aibar/Oibar",
			'31011' => "Allín/Allin",
			'31012' => "Allo",
			'31010' => "Altsasu/Alsasua",
			'31013' => "Améscoa Baja",
			'31014' => "Ancín/Antzin",
			'31015' => "Andosilla",
			'31016' => "Ansoáin/Antsoain",
			'31017' => "Anue",
			'31018' => "Añorbe",
			'31019' => "Aoiz/Agoitz",
			'31020' => "Araitz",
			'31025' => "Arakil",
			'31021' => "Aranarache/Aranaratxe",
			'31023' => "Aranguren",
			'31024' => "Arano",
			'31022' => "Arantza",
			'31026' => "Aras",
			'31027' => "Arbizu",
			'31028' => "Arce/Artzi",
			'31029' => "Arcos, Los",
			'31030' => "Arellano",
			'31031' => "Areso",
			'31032' => "Arguedas",
			'31033' => "Aria",
			'31034' => "Aribe",
			'31035' => "Armañanzas",
			'31036' => "Arróniz",
			'31037' => "Arruazu",
			'31038' => "Artajona",
			'31039' => "Artazu",
			'31040' => "Atetz",
			'31058' => "Auritz/Burguete",
			'31041' => "Ayegui/Aiegi",
			'31042' => "Azagra",
			'31043' => "Azuelo",
			'31044' => "Bakaiku",
			'31901' => "Barañáin/Barañain",
			'31045' => "Barásoain",
			'31046' => "Barbarin",
			'31047' => "Bargota",
			'31048' => "Barillas",
			'31049' => "Basaburua",
			'31050' => "Baztan",
			'31137' => "Beintza-Labaien",
			'31051' => "Beire",
			'31052' => "Belascoáin",
			'31250' => "Bera",
			'31053' => "Berbinzana",
			'31905' => "Beriáin",
			'31902' => "Berrioplano/Berriobeiti",
			'31903' => "Berriozar",
			'31054' => "Bertizarana",
			'31055' => "Betelu",
			'31253' => "Bidaurreta",
			'31056' => "Biurrun-Olcoz",
			'31057' => "Buñuel",
			'31059' => "Burgui/Burgi",
			'31060' => "Burlada/Burlata",
			'31061' => "Busto, El",
			'31062' => "Cabanillas",
			'31063' => "Cabredo",
			'31064' => "Cadreita",
			'31065' => "Caparroso",
			'31066' => "Cárcar",
			'31067' => "Carcastillo",
			'31068' => "Cascante",
			'31069' => "Cáseda",
			'31070' => "Castejón",
			'31071' => "Castillonuevo",
			'31193' => "Cendea de Olza/Oltza Zendea",
			'31072' => "Cintruénigo",
			'31074' => "Cirauqui/Zirauki",
			'31075' => "Ciriza/Ziritza",
			'31076' => "Cizur",
			'31077' => "Corella",
			'31078' => "Cortes",
			'31079' => "Desojo",
			'31080' => "Dicastillo",
			'31081' => "Donamaria",
			'31221' => "Doneztebe/Santesteban",
			'31083' => "Echarri/Etxarri",
			'31087' => "Elgorriaga",
			'31089' => "Enériz/Eneritz",
			'31090' => "Eratsun",
			'31091' => "Ergoiena",
			'31092' => "Erro",
			'31094' => "Eslava",
			'31095' => "Esparza de Salazar/Espartza Zaraitzu",
			'31096' => "Espronceda",
			'31097' => "Estella-Lizarra",
			'31098' => "Esteribar",
			'31099' => "Etayo",
			'31082' => "Etxalar",
			'31084' => "Etxarri Aranatz",
			'31085' => "Etxauri",
			'31100' => "Eulate",
			'31101' => "Ezcabarte",
			'31093' => "Ezcároz/Ezkaroze",
			'31102' => "Ezkurra",
			'31103' => "Ezprogui",
			'31104' => "Falces",
			'31105' => "Fitero",
			'31106' => "Fontellas",
			'31107' => "Funes",
			'31108' => "Fustiñana",
			'31109' => "Galar",
			'31110' => "Gallipienzo/Galipentzu",
			'31111' => "Gallués/Galoze",
			'31112' => "Garaioa",
			'31113' => "Garde",
			'31114' => "Garínoain",
			'31115' => "Garralda",
			'31116' => "Genevilla",
			'31117' => "Goizueta",
			'31118' => "Goñi",
			'31119' => "Güesa/Gorza",
			'31120' => "Guesálaz/Gesalatz",
			'31121' => "Guirguillano",
			'31256' => "Hiriberri/Villanueva de Aezkoa",
			'31122' => "Huarte/Uharte",
			'31124' => "Ibargoiti",
			'31259' => "Igantzi",
			'31125' => "Igúzquiza",
			'31126' => "Imotz",
			'31127' => "Irañeta",
			'31904' => "Irurtzun",
			'31128' => "Isaba/Izaba",
			'31129' => "Ituren",
			'31130' => "Iturmendi",
			'31131' => "Iza/Itza",
			'31132' => "Izagaondoa",
			'31133' => "Izalzu/Itzaltzu",
			'31134' => "Jaurrieta",
			'31135' => "Javier",
			'31136' => "Juslapeña/Txulapain",
			'31138' => "Lakuntza",
			'31139' => "Lana",
			'31140' => "Lantz",
			'31141' => "Lapoblación",
			'31142' => "Larraga",
			'31143' => "Larraona",
			'31144' => "Larraun",
			'31145' => "Lazagurría",
			'31146' => "Leache/Leatxe",
			'31147' => "Legarda",
			'31148' => "Legaria",
			'31149' => "Leitza",
			'31908' => "Lekunberri",
			'31150' => "Leoz/Leotz",
			'31151' => "Lerga",
			'31152' => "Lerín",
			'31153' => "Lesaka",
			'31154' => "Lezaun",
			'31155' => "Liédena",
			'31156' => "Lizoain-Arriasgoiti/Lizoainibar-Arriasgoiti",
			'31157' => "Lodosa",
			'31158' => "Lónguida/Longida",
			'31159' => "Lumbier",
			'31160' => "Luquin",
			'31248' => "Luzaide/Valcarlos",
			'31161' => "Mañeru",
			'31162' => "Marañón",
			'31163' => "Marcilla",
			'31164' => "Mélida",
			'31165' => "Mendavia",
			'31166' => "Mendaza",
			'31167' => "Mendigorria",
			'31168' => "Metauten",
			'31169' => "Milagro",
			'31170' => "Mirafuentes",
			'31171' => "Miranda de Arga",
			'31172' => "Monreal/Elo",
			'31173' => "Monteagudo",
			'31174' => "Morentin",
			'31175' => "Mues",
			'31176' => "Murchante",
			'31177' => "Murieta",
			'31178' => "Murillo el Cuende",
			'31179' => "Murillo el Fruto",
			'31180' => "Muruzábal",
			'31181' => "Navascués/Nabaskoze",
			'31182' => "Nazar",
			'31088' => "Noáin (Valle de Elorz)/Noain (Elortzibar)",
			'31183' => "Obanos",
			'31185' => "Ochagavía/Otsagabia",
			'31184' => "Oco",
			'31186' => "Odieta",
			'31187' => "Oiz",
			'31188' => "Olaibar",
			'31189' => "Olazti/Olazagutía",
			'31190' => "Olejua",
			'31191' => "Olite/Erriberri",
			'31192' => "Olóriz/Oloritz",
			'31195' => "Orbaizeta",
			'31196' => "Orbara",
			'31197' => "Orísoain",
			'31906' => "Orkoien",
			'31198' => "Oronz/Orontze",
			'31199' => "Oroz-Betelu/Orotz-Betelu",
			'31211' => "Orreaga/Roncesvalles",
			'31200' => "Oteiza",
			'31201' => "Pamplona/Iruña",
			'31202' => "Peralta/Azkoien",
			'31203' => "Petilla de Aragón",
			'31204' => "Piedramillera",
			'31205' => "Pitillas",
			'31206' => "Puente la Reina/Gares",
			'31207' => "Pueyo/Puiu",
			'31208' => "Ribaforada",
			'31209' => "Romanzado/Erromantzatua",
			'31210' => "Roncal/Erronkari",
			'31212' => "Sada",
			'31213' => "Saldías",
			'31214' => "Salinas de Oro/Jaitz",
			'31215' => "San Adrián",
			'31217' => "San Martín de Unx",
			'31216' => "Sangüesa/Zangoza",
			'31219' => "Sansol",
			'31220' => "Santacara",
			'31222' => "Sarriés/Sartze",
			'31223' => "Sartaguda",
			'31224' => "Sesma",
			'31225' => "Sorlada",
			'31226' => "Sunbilla",
			'31227' => "Tafalla",
			'31228' => "Tiebas-Muruarte de Reta",
			'31229' => "Tirapu",
			'31230' => "Torralba del Río",
			'31231' => "Torres del Río",
			'31232' => "Tudela",
			'31233' => "Tulebras",
			'31234' => "Ucar",
			'31123' => "Uharte Arakil",
			'31235' => "Ujué/Uxue",
			'31236' => "Ultzama",
			'31237' => "Unciti",
			'31238' => "Unzué/Untzue",
			'31239' => "Urdazubi/Urdax",
			'31240' => "Urdiain",
			'31241' => "Urraul Alto",
			'31242' => "Urraul Bajo",
			'31244' => "Urroz",
			'31243' => "Urroz-Villa",
			'31245' => "Urzainqui/Urzainki",
			'31246' => "Uterga",
			'31247' => "Uztárroz/Uztarroze",
			'31086' => "Valle de Egüés/Eguesibar",
			'31194' => "Valle de Ollo/Ollaran",
			'31260' => "Valle de Yerri/Deierri",
			'31249' => "Valtierra",
			'31251' => "Viana",
			'31252' => "Vidángoz/Bidankoze",
			'31254' => "Villafranca",
			'31255' => "Villamayor de Monjardín",
			'31257' => "Villatuerta",
			'31258' => "Villava/Atarrabia",
			'31261' => "Yesa",
			'31262' => "Zabalza/Zabaltza",
			'31073' => "Ziordia",
			'31907' => "Zizur Mayor/Zizur Nagusia",
			'31263' => "Zubieta",
			'31264' => "Zugarramurdi",
			'31265' => "Zúñiga",
			'32001' => "Allariz",
			'32002' => "Amoeiro",
			'32003' => "Arnoia, A",
			'32004' => "Avión",
			'32005' => "Baltar",
			'32006' => "Bande",
			'32007' => "Baños de Molgas",
			'32008' => "Barbadás",
			'32009' => "Barco de Valdeorras, O",
			'32010' => "Beade",
			'32011' => "Beariz",
			'32012' => "Blancos, Os",
			'32013' => "Boborás",
			'32014' => "Bola, A",
			'32015' => "Bolo, O",
			'32016' => "Calvos de Randín",
			'32018' => "Carballeda de Avia",
			'32017' => "Carballeda de Valdeorras",
			'32019' => "Carballiño, O",
			'32020' => "Cartelle",
			'32022' => "Castrelo de Miño",
			'32021' => "Castrelo do Val",
			'32023' => "Castro Caldelas",
			'32024' => "Celanova",
			'32025' => "Cenlle",
			'32029' => "Chandrexa de Queixa",
			'32026' => "Coles",
			'32027' => "Cortegada",
			'32028' => "Cualedro",
			'32030' => "Entrimo",
			'32031' => "Esgos",
			'32033' => "Gomesende",
			'32034' => "Gudiña, A",
			'32035' => "Irixo, O",
			'32038' => "Larouco",
			'32039' => "Laza",
			'32040' => "Leiro",
			'32041' => "Lobeira",
			'32042' => "Lobios",
			'32043' => "Maceda",
			'32044' => "Manzaneda",
			'32045' => "Maside",
			'32046' => "Melón",
			'32047' => "Merca, A",
			'32048' => "Mezquita, A",
			'32049' => "Montederramo",
			'32050' => "Monterrei",
			'32051' => "Muíños",
			'32052' => "Nogueira de Ramuín",
			'32053' => "Oímbra",
			'32054' => "Ourense",
			'32055' => "Paderne de Allariz",
			'32056' => "Padrenda",
			'32057' => "Parada de Sil",
			'32058' => "Pereiro de Aguiar, O",
			'32059' => "Peroxa, A",
			'32060' => "Petín",
			'32061' => "Piñor",
			'32063' => "Pobra de Trives, A",
			'32064' => "Pontedeva",
			'32062' => "Porqueira",
			'32065' => "Punxín",
			'32066' => "Quintela de Leirado",
			'32067' => "Rairiz de Veiga",
			'32068' => "Ramirás",
			'32069' => "Ribadavia",
			'32071' => "Riós",
			'32072' => "Rúa, A",
			'32073' => "Rubiá",
			'32074' => "San Amaro",
			'32075' => "San Cibrao das Viñas",
			'32076' => "San Cristovo de Cea",
			'32070' => "San Xoán de Río",
			'32077' => "Sandiás",
			'32078' => "Sarreaus",
			'32079' => "Taboadela",
			'32080' => "Teixeira, A",
			'32081' => "Toén",
			'32082' => "Trasmiras",
			'32083' => "Veiga, A",
			'32084' => "Verea",
			'32085' => "Verín",
			'32086' => "Viana do Bolo",
			'32087' => "Vilamarín",
			'32088' => "Vilamartín de Valdeorras",
			'32089' => "Vilar de Barrio",
			'32090' => "Vilar de Santos",
			'32091' => "Vilardevós",
			'32092' => "Vilariño de Conso",
			'32032' => "Xinzo de Limia",
			'32036' => "Xunqueira de Ambía",
			'32037' => "Xunqueira de Espadanedo",
			'33001' => "Allande",
			'33002' => "Aller",
			'33003' => "Amieva",
			'33004' => "Avilés",
			'33005' => "Belmonte de Miranda",
			'33006' => "Bimenes",
			'33007' => "Boal",
			'33008' => "Cabrales",
			'33009' => "Cabranes",
			'33010' => "Candamo",
			'33012' => "Cangas de Onís",
			'33011' => "Cangas del Narcea",
			'33013' => "Caravia",
			'33014' => "Carreño",
			'33015' => "Caso",
			'33016' => "Castrillón",
			'33017' => "Castropol",
			'33018' => "Coaña",
			'33019' => "Colunga",
			'33020' => "Corvera de Asturias",
			'33021' => "Cudillero",
			'33022' => "Degaña",
			'33023' => "Franco, El",
			'33024' => "Gijón",
			'33025' => "Gozón",
			'33026' => "Grado",
			'33027' => "Grandas de Salime",
			'33028' => "Ibias",
			'33029' => "Illano",
			'33030' => "Illas",
			'33031' => "Langreo",
			'33032' => "Laviana",
			'33033' => "Lena",
			'33035' => "Llanera",
			'33036' => "Llanes",
			'33037' => "Mieres",
			'33038' => "Morcín",
			'33039' => "Muros de Nalón",
			'33040' => "Nava",
			'33041' => "Navia",
			'33042' => "Noreña",
			'33043' => "Onís",
			'33044' => "Oviedo",
			'33045' => "Parres",
			'33046' => "Peñamellera Alta",
			'33047' => "Peñamellera Baja",
			'33048' => "Pesoz",
			'33049' => "Piloña",
			'33050' => "Ponga",
			'33051' => "Pravia",
			'33052' => "Proaza",
			'33053' => "Quirós",
			'33054' => "Regueras, Las",
			'33055' => "Ribadedeva",
			'33056' => "Ribadesella",
			'33057' => "Ribera de Arriba",
			'33058' => "Riosa",
			'33059' => "Salas",
			'33061' => "San Martín de Oscos",
			'33060' => "San Martín del Rey Aurelio",
			'33063' => "San Tirso de Abres",
			'33062' => "Santa Eulalia de Oscos",
			'33064' => "Santo Adriano",
			'33065' => "Sariego",
			'33066' => "Siero",
			'33067' => "Sobrescobio",
			'33068' => "Somiedo",
			'33069' => "Soto del Barco",
			'33070' => "Tapia de Casariego",
			'33071' => "Taramundi",
			'33072' => "Teverga",
			'33073' => "Tineo",
			'33034' => "Valdés",
			'33074' => "Vegadeo",
			'33075' => "Villanueva de Oscos",
			'33076' => "Villaviciosa",
			'33077' => "Villayón",
			'33078' => "Yernes y Tameza",
			'34001' => "Abarca de Campos",
			'34003' => "Abia de las Torres",
			'34004' => "Aguilar de Campoo",
			'34005' => "Alar del Rey",
			'34006' => "Alba de Cerrato",
			'34009' => "Amayuelas de Arriba",
			'34010' => "Ampudia",
			'34011' => "Amusco",
			'34012' => "Antigüedad",
			'34015' => "Arconada",
			'34017' => "Astudillo",
			'34018' => "Autilla del Pino",
			'34019' => "Autillo de Campos",
			'34020' => "Ayuela",
			'34022' => "Baltanás",
			'34024' => "Baquerín de Campos",
			'34025' => "Bárcena de Campos",
			'34027' => "Barruelo de Santullán",
			'34028' => "Báscones de Ojeda",
			'34029' => "Becerril de Campos",
			'34031' => "Belmonte de Campos",
			'34032' => "Berzosilla",
			'34033' => "Boada de Campos",
			'34035' => "Boadilla de Rioseco",
			'34034' => "Boadilla del Camino",
			'34036' => "Brañosera",
			'34037' => "Buenavista de Valdavia",
			'34038' => "Bustillo de la Vega",
			'34039' => "Bustillo del Páramo de Carrión",
			'34041' => "Calahorra de Boedo",
			'34042' => "Calzada de los Molinos",
			'34045' => "Capillas",
			'34046' => "Cardeñosa de Volpejera",
			'34047' => "Carrión de los Condes",
			'34048' => "Castil de Vela",
			'34049' => "Castrejón de la Peña",
			'34050' => "Castrillo de Don Juan",
			'34051' => "Castrillo de Onielo",
			'34052' => "Castrillo de Villavega",
			'34053' => "Castromocho",
			'34055' => "Cervatos de la Cueza",
			'34056' => "Cervera de Pisuerga",
			'34057' => "Cevico de la Torre",
			'34058' => "Cevico Navero",
			'34059' => "Cisneros",
			'34060' => "Cobos de Cerrato",
			'34061' => "Collazos de Boedo",
			'34062' => "Congosto de Valdavia",
			'34063' => "Cordovilla la Real",
			'34066' => "Cubillas de Cerrato",
			'34067' => "Dehesa de Montejo",
			'34068' => "Dehesa de Romanos",
			'34069' => "Dueñas",
			'34070' => "Espinosa de Cerrato",
			'34071' => "Espinosa de Villagonzalo",
			'34072' => "Frechilla",
			'34073' => "Fresno del Río",
			'34074' => "Frómista",
			'34076' => "Fuentes de Nava",
			'34077' => "Fuentes de Valdepero",
			'34079' => "Grijota",
			'34080' => "Guardo",
			'34081' => "Guaza de Campos",
			'34082' => "Hérmedes de Cerrato",
			'34083' => "Herrera de Pisuerga",
			'34084' => "Herrera de Valdecañas",
			'34086' => "Hontoria de Cerrato",
			'34087' => "Hornillos de Cerrato",
			'34088' => "Husillos",
			'34089' => "Itero de la Vega",
			'34091' => "Lagartos",
			'34092' => "Lantadilla",
			'34094' => "Ledigos",
			'34903' => "Loma de Ucieza",
			'34096' => "Lomas",
			'34098' => "Magaz de Pisuerga",
			'34099' => "Manquillos",
			'34100' => "Mantinos",
			'34101' => "Marcilla de Campos",
			'34102' => "Mazariegos",
			'34103' => "Mazuecos de Valdeginate",
			'34104' => "Melgar de Yuso",
			'34106' => "Meneses de Campos",
			'34107' => "Micieces de Ojeda",
			'34108' => "Monzón de Campos",
			'34109' => "Moratinos",
			'34110' => "Mudá",
			'34112' => "Nogal de las Huertas",
			'34113' => "Olea de Boedo",
			'34114' => "Olmos de Ojeda",
			'34116' => "Osornillo",
			'34901' => "Osorno la Mayor",
			'34120' => "Palencia",
			'34121' => "Palenzuela",
			'34122' => "Páramo de Boedo",
			'34123' => "Paredes de Nava",
			'34124' => "Payo de Ojeda",
			'34125' => "Pedraza de Campos",
			'34126' => "Pedrosa de la Vega",
			'34127' => "Perales",
			'34904' => "Pernía, La",
			'34129' => "Pino del Río",
			'34130' => "Piña de Campos",
			'34131' => "Población de Arroyo",
			'34132' => "Población de Campos",
			'34133' => "Población de Cerrato",
			'34134' => "Polentinos",
			'34135' => "Pomar de Valdivia",
			'34136' => "Poza de la Vega",
			'34137' => "Pozo de Urama",
			'34139' => "Prádanos de Ojeda",
			'34140' => "Puebla de Valdavia, La",
			'34141' => "Quintana del Puente",
			'34143' => "Quintanilla de Onsoña",
			'34146' => "Reinoso de Cerrato",
			'34147' => "Renedo de la Vega",
			'34149' => "Requena de Campos",
			'34151' => "Respenda de la Peña",
			'34152' => "Revenga de Campos",
			'34154' => "Revilla de Collazos",
			'34155' => "Ribas de Campos",
			'34156' => "Riberos de la Cueza",
			'34157' => "Saldaña",
			'34158' => "Salinas de Pisuerga",
			'34159' => "San Cebrián de Campos",
			'34160' => "San Cebrián de Mudá",
			'34161' => "San Cristóbal de Boedo",
			'34163' => "San Mamés de Campos",
			'34165' => "San Román de la Cuba",
			'34167' => "Santa Cecilia del Alcor",
			'34168' => "Santa Cruz de Boedo",
			'34169' => "Santervás de la Vega",
			'34170' => "Santibáñez de Ecla",
			'34171' => "Santibáñez de la Peña",
			'34174' => "Santoyo",
			'34175' => "Serna, La",
			'34177' => "Soto de Cerrato",
			'34176' => "Sotobañado y Priorato",
			'34178' => "Tabanera de Cerrato",
			'34179' => "Tabanera de Valdavia",
			'34180' => "Támara de Campos",
			'34181' => "Tariego de Cerrato",
			'34182' => "Torquemada",
			'34184' => "Torremormojón",
			'34185' => "Triollo",
			'34186' => "Valbuena de Pisuerga",
			'34189' => "Valdeolmillos",
			'34190' => "Valderrábano",
			'34192' => "Valde-Ucieza",
			'34196' => "Valle de Cerrato",
			'34902' => "Valle del Retortillo",
			'34199' => "Velilla del Río Carrión",
			'34023' => "Venta de Baños",
			'34201' => "Vertavillo",
			'34093' => "Vid de Ojeda, La",
			'34202' => "Villabasta de Valdavia",
			'34204' => "Villacidaler",
			'34205' => "Villaconancio",
			'34206' => "Villada",
			'34208' => "Villaeles de Valdavia",
			'34210' => "Villahán",
			'34211' => "Villaherreros",
			'34213' => "Villalaco",
			'34214' => "Villalba de Guardo",
			'34215' => "Villalcázar de Sirga",
			'34216' => "Villalcón",
			'34217' => "Villalobón",
			'34218' => "Villaluenga de la Vega",
			'34220' => "Villamartín de Campos",
			'34221' => "Villamediana",
			'34222' => "Villameriel",
			'34223' => "Villamoronta",
			'34224' => "Villamuera de la Cueza",
			'34225' => "Villamuriel de Cerrato",
			'34227' => "Villanueva del Rebollar",
			'34228' => "Villanuño de Valdavia",
			'34229' => "Villaprovedo",
			'34230' => "Villarmentero de Campos",
			'34231' => "Villarrabé",
			'34232' => "Villarramiel",
			'34233' => "Villasarracino",
			'34234' => "Villasila de Valdavia",
			'34236' => "Villaturde",
			'34237' => "Villaumbrales",
			'34238' => "Villaviudas",
			'34240' => "Villerías de Campos",
			'34241' => "Villodre",
			'34242' => "Villodrigo",
			'34243' => "Villoldo",
			'34245' => "Villota del Páramo",
			'34246' => "Villovieco",
			'35001' => "Agaete",
			'35002' => "Agüimes",
			'35020' => "Aldea de San Nicolás, La",
			'35003' => "Antigua",
			'35004' => "Arrecife",
			'35005' => "Artenara",
			'35006' => "Arucas",
			'35007' => "Betancuria",
			'35008' => "Firgas",
			'35009' => "Gáldar",
			'35010' => "Haría",
			'35011' => "Ingenio",
			'35012' => "Mogán",
			'35013' => "Moya",
			'35014' => "Oliva, La",
			'35015' => "Pájara",
			'35016' => "Palmas de Gran Canaria, Las",
			'35017' => "Puerto del Rosario",
			'35018' => "San Bartolomé",
			'35019' => "San Bartolomé de Tirajana",
			'35021' => "Santa Brígida",
			'35022' => "Santa Lucía de Tirajana",
			'35023' => "Santa María de Guía de Gran Canaria",
			'35024' => "Teguise",
			'35025' => "Tejeda",
			'35026' => "Telde",
			'35027' => "Teror",
			'35028' => "Tías",
			'35029' => "Tinajo",
			'35030' => "Tuineje",
			'35032' => "Valleseco",
			'35031' => "Valsequillo de Gran Canaria",
			'35033' => "Vega de San Mateo",
			'35034' => "Yaiza",
			'36020' => "Agolada",
			'36001' => "Arbo",
			'36003' => "Baiona",
			'36002' => "Barro",
			'36004' => "Bueu",
			'36005' => "Caldas de Reis",
			'36006' => "Cambados",
			'36007' => "Campo Lameiro",
			'36008' => "Cangas",
			'36009' => "Cañiza, A",
			'36010' => "Catoira",
			'36902' => "Cerdedo-Cotobade",
			'36013' => "Covelo",
			'36014' => "Crecente",
			'36015' => "Cuntis",
			'36016' => "Dozón",
			'36017' => "Estrada, A",
			'36018' => "Forcarei",
			'36019' => "Fornelos de Montes",
			'36021' => "Gondomar",
			'36022' => "Grove, O",
			'36023' => "Guarda, A",
			'36901' => "Illa de Arousa, A",
			'36024' => "Lalín",
			'36025' => "Lama, A",
			'36026' => "Marín",
			'36027' => "Meaño",
			'36028' => "Meis",
			'36029' => "Moaña",
			'36030' => "Mondariz",
			'36031' => "Mondariz-Balneario",
			'36032' => "Moraña",
			'36033' => "Mos",
			'36034' => "Neves, As",
			'36035' => "Nigrán",
			'36036' => "Oia",
			'36037' => "Pazos de Borbén",
			'36041' => "Poio",
			'36043' => "Ponte Caldelas",
			'36042' => "Ponteareas",
			'36044' => "Pontecesures",
			'36038' => "Pontevedra",
			'36039' => "Porriño, O",
			'36040' => "Portas",
			'36045' => "Redondela",
			'36046' => "Ribadumia",
			'36047' => "Rodeiro",
			'36048' => "Rosal, O",
			'36049' => "Salceda de Caselas",
			'36050' => "Salvaterra de Miño",
			'36051' => "Sanxenxo",
			'36052' => "Silleda",
			'36053' => "Soutomaior",
			'36054' => "Tomiño",
			'36055' => "Tui",
			'36056' => "Valga",
			'36057' => "Vigo",
			'36059' => "Vila de Cruces",
			'36058' => "Vilaboa",
			'36060' => "Vilagarcía de Arousa",
			'36061' => "Vilanova de Arousa",
			'37001' => "Abusejo",
			'37002' => "Agallas",
			'37003' => "Ahigal de los Aceiteros",
			'37004' => "Ahigal de Villarino",
			'37005' => "Alameda de Gardón, La",
			'37006' => "Alamedilla, La",
			'37007' => "Alaraz",
			'37008' => "Alba de Tormes",
			'37009' => "Alba de Yeltes",
			'37010' => "Alberca, La",
			'37011' => "Alberguería de Argañán, La",
			'37012' => "Alconada",
			'37015' => "Aldea del Obispo",
			'37013' => "Aldeacipreste",
			'37014' => "Aldeadávila de la Ribera",
			'37016' => "Aldealengua",
			'37017' => "Aldeanueva de Figueroa",
			'37018' => "Aldeanueva de la Sierra",
			'37019' => "Aldearrodrigo",
			'37020' => "Aldearrubia",
			'37021' => "Aldeaseca de Alba",
			'37022' => "Aldeaseca de la Frontera",
			'37023' => "Aldeatejada",
			'37024' => "Aldeavieja de Tormes",
			'37025' => "Aldehuela de la Bóveda",
			'37026' => "Aldehuela de Yeltes",
			'37027' => "Almenara de Tormes",
			'37028' => "Almendra",
			'37029' => "Anaya de Alba",
			'37030' => "Añover de Tormes",
			'37031' => "Arabayona de Mógica",
			'37032' => "Arapiles",
			'37033' => "Arcediano",
			'37034' => "Arco, El",
			'37035' => "Armenteros",
			'37037' => "Atalaya, La",
			'37038' => "Babilafuente",
			'37039' => "Bañobárez",
			'37040' => "Barbadillo",
			'37041' => "Barbalos",
			'37042' => "Barceo",
			'37044' => "Barruecopardo",
			'37045' => "Bastida, La",
			'37046' => "Béjar",
			'37047' => "Beleña",
			'37049' => "Bermellar",
			'37050' => "Berrocal de Huebra",
			'37051' => "Berrocal de Salvatierra",
			'37052' => "Boada",
			'37054' => "Bodón, El",
			'37055' => "Bogajo",
			'37056' => "Bouza, La",
			'37057' => "Bóveda del Río Almar",
			'37058' => "Brincones",
			'37059' => "Buenamadre",
			'37060' => "Buenavista",
			'37061' => "Cabaco, El",
			'37063' => "Cabeza de Béjar, La",
			'37065' => "Cabeza del Caballo",
			'37062' => "Cabezabellosa de la Calzada",
			'37067' => "Cabrerizos",
			'37068' => "Cabrillas",
			'37069' => "Calvarrasa de Abajo",
			'37070' => "Calvarrasa de Arriba",
			'37071' => "Calzada de Béjar, La",
			'37072' => "Calzada de Don Diego",
			'37073' => "Calzada de Valdunciel",
			'37074' => "Campillo de Azaba",
			'37077' => "Campo de Peñaranda, El",
			'37078' => "Candelario",
			'37079' => "Canillas de Abajo",
			'37080' => "Cantagallo",
			'37081' => "Cantalapiedra",
			'37082' => "Cantalpino",
			'37083' => "Cantaracillo",
			'37085' => "Carbajosa de la Sagrada",
			'37086' => "Carpio de Azaba",
			'37087' => "Carrascal de Barregas",
			'37088' => "Carrascal del Obispo",
			'37089' => "Casafranca",
			'37090' => "Casas del Conde, Las",
			'37091' => "Casillas de Flores",
			'37092' => "Castellanos de Moriscos",
			'37185' => "Castellanos de Villiquera",
			'37096' => "Castillejo de Martín Viejo",
			'37097' => "Castraz",
			'37098' => "Cepeda",
			'37099' => "Cereceda de la Sierra",
			'37100' => "Cerezal de Peñahorcada",
			'37101' => "Cerralbo",
			'37102' => "Cerro, El",
			'37103' => "Cespedosa de Tormes",
			'37114' => "Chagarcía Medianero",
			'37104' => "Cilleros de la Bastida",
			'37106' => "Cipérez",
			'37107' => "Ciudad Rodrigo",
			'37108' => "Coca de Alba",
			'37109' => "Colmenar de Montemayor",
			'37110' => "Cordovilla",
			'37112' => "Cristóbal",
			'37113' => "Cubo de Don Sancho, El",
			'37115' => "Dios le Guarde",
			'37116' => "Doñinos de Ledesma",
			'37117' => "Doñinos de Salamanca",
			'37118' => "Éjeme",
			'37120' => "Encina de San Silvestre",
			'37119' => "Encina, La",
			'37121' => "Encinas de Abajo",
			'37122' => "Encinas de Arriba",
			'37123' => "Encinasola de los Comendadores",
			'37124' => "Endrinal",
			'37125' => "Escurial de la Sierra",
			'37126' => "Espadaña",
			'37127' => "Espeja",
			'37128' => "Espino de la Orbada",
			'37129' => "Florida de Liébana",
			'37130' => "Forfoleda",
			'37131' => "Frades de la Sierra",
			'37132' => "Fregeneda, La",
			'37133' => "Fresnedoso",
			'37134' => "Fresno Alhándiga",
			'37135' => "Fuente de San Esteban, La",
			'37136' => "Fuenteguinaldo",
			'37137' => "Fuenteliante",
			'37138' => "Fuenterroble de Salvatierra",
			'37139' => "Fuentes de Béjar",
			'37140' => "Fuentes de Oñoro",
			'37141' => "Gajates",
			'37142' => "Galindo y Perahuy",
			'37143' => "Galinduste",
			'37144' => "Galisancho",
			'37145' => "Gallegos de Argañán",
			'37146' => "Gallegos de Solmirón",
			'37147' => "Garcibuey",
			'37148' => "Garcihernández",
			'37149' => "Garcirrey",
			'37150' => "Gejuelo del Barro",
			'37151' => "Golpejas",
			'37152' => "Gomecello",
			'37154' => "Guadramiro",
			'37155' => "Guijo de Ávila",
			'37156' => "Guijuelo",
			'37157' => "Herguijuela de Ciudad Rodrigo",
			'37158' => "Herguijuela de la Sierra",
			'37159' => "Herguijuela del Campo",
			'37160' => "Hinojosa de Duero",
			'37161' => "Horcajo de Montemayor",
			'37162' => "Horcajo Medianero",
			'37163' => "Hoya, La",
			'37164' => "Huerta",
			'37165' => "Iruelos",
			'37166' => "Ituero de Azaba",
			'37167' => "Juzbado",
			'37168' => "Lagunilla",
			'37169' => "Larrodrigo",
			'37170' => "Ledesma",
			'37171' => "Ledrada",
			'37172' => "Linares de Riofrío",
			'37173' => "Lumbrales",
			'37175' => "Machacón",
			'37174' => "Macotera",
			'37176' => "Madroñal",
			'37177' => "Maíllo, El",
			'37178' => "Malpartida",
			'37179' => "Mancera de Abajo",
			'37180' => "Manzano, El",
			'37181' => "Martiago",
			'37183' => "Martín de Yeltes",
			'37182' => "Martinamor",
			'37184' => "Masueco",
			'37186' => "Mata de Ledesma, La",
			'37187' => "Matilla de los Caños del Río",
			'37188' => "Maya, La",
			'37189' => "Membribe de la Sierra",
			'37190' => "Mieza",
			'37191' => "Milano, El",
			'37192' => "Miranda de Azán",
			'37193' => "Miranda del Castañar",
			'37194' => "Mogarraz",
			'37195' => "Molinillo",
			'37196' => "Monforte de la Sierra",
			'37197' => "Monleón",
			'37198' => "Monleras",
			'37199' => "Monsagro",
			'37200' => "Montejo",
			'37201' => "Montemayor del Río",
			'37202' => "Monterrubio de Armuña",
			'37203' => "Monterrubio de la Sierra",
			'37204' => "Morasverdes",
			'37205' => "Morille",
			'37206' => "Moríñigo",
			'37207' => "Moriscos",
			'37208' => "Moronta",
			'37209' => "Mozárbez",
			'37211' => "Narros de Matalayegua",
			'37213' => "Nava de Béjar",
			'37214' => "Nava de Francia",
			'37215' => "Nava de Sotrobal",
			'37212' => "Navacarros",
			'37216' => "Navales",
			'37217' => "Navalmoral de Béjar",
			'37218' => "Navamorales",
			'37219' => "Navarredonda de la Rinconada",
			'37221' => "Navasfrías",
			'37222' => "Negrilla de Palencia",
			'37223' => "Olmedo de Camaces",
			'37224' => "Orbada, La",
			'37225' => "Pajares de la Laguna",
			'37226' => "Palacios del Arzobispo",
			'37228' => "Palaciosrubios",
			'37229' => "Palencia de Negrilla",
			'37230' => "Parada de Arriba",
			'37231' => "Parada de Rubiales",
			'37232' => "Paradinas de San Juan",
			'37233' => "Pastores",
			'37234' => "Payo, El",
			'37235' => "Pedraza de Alba",
			'37236' => "Pedrosillo de Alba",
			'37237' => "Pedrosillo de los Aires",
			'37238' => "Pedrosillo el Ralo",
			'37239' => "Pedroso de la Armuña, El",
			'37240' => "Pelabravo",
			'37241' => "Pelarrodríguez",
			'37242' => "Pelayos",
			'37243' => "Peña, La",
			'37244' => "Peñacaballera",
			'37245' => "Peñaparda",
			'37246' => "Peñaranda de Bracamonte",
			'37247' => "Peñarandilla",
			'37248' => "Peralejos de Abajo",
			'37249' => "Peralejos de Arriba",
			'37250' => "Pereña de la Ribera",
			'37251' => "Peromingo",
			'37252' => "Pinedas",
			'37253' => "Pino de Tormes, El",
			'37254' => "Pitiegua",
			'37255' => "Pizarral",
			'37256' => "Poveda de las Cintas",
			'37257' => "Pozos de Hinojo",
			'37258' => "Puebla de Azaba",
			'37259' => "Puebla de San Medel",
			'37260' => "Puebla de Yeltes",
			'37261' => "Puente del Congosto",
			'37262' => "Puertas",
			'37263' => "Puerto de Béjar",
			'37264' => "Puerto Seguro",
			'37265' => "Rágama",
			'37266' => "Redonda, La",
			'37267' => "Retortillo",
			'37268' => "Rinconada de la Sierra, La",
			'37269' => "Robleda",
			'37270' => "Robliza de Cojos",
			'37271' => "Rollán",
			'37272' => "Saelices el Chico",
			'37273' => "Sagrada, La",
			'37303' => "Sahugo, El",
			'37274' => "Salamanca",
			'37275' => "Saldeana",
			'37276' => "Salmoral",
			'37277' => "Salvatierra de Tormes",
			'37278' => "San Cristóbal de la Cuesta",
			'37284' => "San Esteban de la Sierra",
			'37285' => "San Felices de los Gallegos",
			'37286' => "San Martín del Castañar",
			'37287' => "San Miguel de Valero",
			'37036' => "San Miguel del Robledo",
			'37288' => "San Morales",
			'37289' => "San Muñoz",
			'37291' => "San Pedro de Rozados",
			'37290' => "San Pedro del Valle",
			'37292' => "San Pelayo de Guareña",
			'37280' => "Sanchón de la Ribera",
			'37281' => "Sanchón de la Sagrada",
			'37282' => "Sanchotello",
			'37279' => "Sancti-Spíritus",
			'37283' => "Sando",
			'37293' => "Santa María de Sando",
			'37294' => "Santa Marta de Tormes",
			'37296' => "Santiago de la Puebla",
			'37297' => "Santibáñez de Béjar",
			'37298' => "Santibáñez de la Sierra",
			'37299' => "Santiz",
			'37300' => "Santos, Los",
			'37301' => "Sardón de los Frailes",
			'37302' => "Saucelle",
			'37304' => "Sepulcro-Hilario",
			'37305' => "Sequeros",
			'37306' => "Serradilla del Arroyo",
			'37307' => "Serradilla del Llano",
			'37309' => "Sierpe, La",
			'37310' => "Sieteiglesias de Tormes",
			'37311' => "Sobradillo",
			'37312' => "Sorihuela",
			'37313' => "Sotoserrano",
			'37314' => "Tabera de Abajo",
			'37315' => "Tala, La",
			'37316' => "Tamames",
			'37317' => "Tarazona de Guareña",
			'37318' => "Tardáguila",
			'37319' => "Tejado, El",
			'37320' => "Tejeda y Segoyuela",
			'37321' => "Tenebrón",
			'37322' => "Terradillos",
			'37323' => "Topas",
			'37324' => "Tordillos",
			'37325' => "Tornadizo, El",
			'37327' => "Torresmenudas",
			'37328' => "Trabanca",
			'37329' => "Tremedal de Tormes",
			'37330' => "Valdecarros",
			'37331' => "Valdefuentes de Sangusín",
			'37332' => "Valdehijaderos",
			'37333' => "Valdelacasa",
			'37334' => "Valdelageve",
			'37335' => "Valdelosa",
			'37336' => "Valdemierque",
			'37337' => "Valderrodrigo",
			'37338' => "Valdunciel",
			'37339' => "Valero",
			'37343' => "Vallejera de Riofrío",
			'37340' => "Valsalabroso",
			'37341' => "Valverde de Valdelacasa",
			'37342' => "Valverdón",
			'37344' => "Vecinos",
			'37345' => "Vega de Tirados",
			'37346' => "Veguillas, Las",
			'37347' => "Vellés, La",
			'37348' => "Ventosa del Río Almar",
			'37349' => "Vídola, La",
			'37351' => "Villaflores",
			'37352' => "Villagonzalo de Tormes",
			'37353' => "Villalba de los Llanos",
			'37354' => "Villamayor",
			'37355' => "Villanueva del Conde",
			'37356' => "Villar de Argañán",
			'37357' => "Villar de Ciervo",
			'37358' => "Villar de Gallimazo",
			'37359' => "Villar de la Yegua",
			'37360' => "Villar de Peralonso",
			'37361' => "Villar de Samaniego",
			'37362' => "Villares de la Reina",
			'37363' => "Villares de Yeltes",
			'37364' => "Villarino de los Aires",
			'37365' => "Villarmayor",
			'37366' => "Villarmuerto",
			'37367' => "Villasbuenas",
			'37368' => "Villasdardo",
			'37369' => "Villaseco de los Gamitos",
			'37370' => "Villaseco de los Reyes",
			'37371' => "Villasrubias",
			'37372' => "Villaverde de Guareña",
			'37373' => "Villavieja de Yeltes",
			'37374' => "Villoria",
			'37375' => "Villoruela",
			'37350' => "Vilvestre",
			'37376' => "Vitigudino",
			'37377' => "Yecla de Yeltes",
			'37378' => "Zamarra",
			'37379' => "Zamayón",
			'37380' => "Zarapicos",
			'37381' => "Zarza de Pumareda, La",
			'37382' => "Zorita de la Frontera",
			'38001' => "Adeje",
			'38002' => "Agulo",
			'38003' => "Alajeró",
			'38004' => "Arafo",
			'38005' => "Arico",
			'38006' => "Arona",
			'38007' => "Barlovento",
			'38008' => "Breña Alta",
			'38009' => "Breña Baja",
			'38010' => "Buenavista del Norte",
			'38011' => "Candelaria",
			'38012' => "Fasnia",
			'38013' => "Frontera",
			'38014' => "Fuencaliente de la Palma",
			'38015' => "Garachico",
			'38016' => "Garafía",
			'38017' => "Granadilla de Abona",
			'38018' => "Guancha, La",
			'38019' => "Guía de Isora",
			'38020' => "Güímar",
			'38021' => "Hermigua",
			'38022' => "Icod de los Vinos",
			'38024' => "Llanos de Aridane, Los",
			'38025' => "Matanza de Acentejo, La",
			'38026' => "Orotava, La",
			'38027' => "Paso, El",
			'38901' => "Pinar de El Hierro, El",
			'38028' => "Puerto de la Cruz",
			'38029' => "Puntagorda",
			'38030' => "Puntallana",
			'38031' => "Realejos, Los",
			'38032' => "Rosario, El",
			'38033' => "San Andrés y Sauces",
			'38023' => "San Cristóbal de La Laguna",
			'38034' => "San Juan de la Rambla",
			'38035' => "San Miguel de Abona",
			'38036' => "San Sebastián de la Gomera",
			'38037' => "Santa Cruz de la Palma",
			'38038' => "Santa Cruz de Tenerife",
			'38039' => "Santa Úrsula",
			'38040' => "Santiago del Teide",
			'38041' => "Sauzal, El",
			'38042' => "Silos, Los",
			'38043' => "Tacoronte",
			'38044' => "Tanque, El",
			'38045' => "Tazacorte",
			'38046' => "Tegueste",
			'38047' => "Tijarafe",
			'38049' => "Valle Gran Rey",
			'38050' => "Vallehermoso",
			'38048' => "Valverde",
			'38051' => "Victoria de Acentejo, La",
			'38052' => "Vilaflor de Chasna",
			'38053' => "Villa de Mazo",
			'39001' => "Alfoz de Lloredo",
			'39002' => "Ampuero",
			'39003' => "Anievas",
			'39004' => "Arenas de Iguña",
			'39005' => "Argoños",
			'39006' => "Arnuero",
			'39007' => "Arredondo",
			'39008' => "Astillero, El",
			'39009' => "Bárcena de Cicero",
			'39010' => "Bárcena de Pie de Concha",
			'39011' => "Bareyo",
			'39012' => "Cabezón de la Sal",
			'39013' => "Cabezón de Liébana",
			'39014' => "Cabuérniga",
			'39015' => "Camaleño",
			'39016' => "Camargo",
			'39027' => "Campoo de Enmedio",
			'39017' => "Campoo de Yuso",
			'39018' => "Cartes",
			'39019' => "Castañeda",
			'39020' => "Castro-Urdiales",
			'39021' => "Cieza",
			'39022' => "Cillorigo de Liébana",
			'39023' => "Colindres",
			'39024' => "Comillas",
			'39025' => "Corrales de Buelna, Los",
			'39026' => "Corvera de Toranzo",
			'39028' => "Entrambasaguas",
			'39029' => "Escalante",
			'39030' => "Guriezo",
			'39031' => "Hazas de Cesto",
			'39032' => "Hermandad de Campoo de Suso",
			'39033' => "Herrerías",
			'39034' => "Lamasón",
			'39035' => "Laredo",
			'39036' => "Liendo",
			'39037' => "Liérganes",
			'39038' => "Limpias",
			'39039' => "Luena",
			'39040' => "Marina de Cudeyo",
			'39041' => "Mazcuerras",
			'39042' => "Medio Cudeyo",
			'39043' => "Meruelo",
			'39044' => "Miengo",
			'39045' => "Miera",
			'39046' => "Molledo",
			'39047' => "Noja",
			'39048' => "Penagos",
			'39049' => "Peñarrubia",
			'39050' => "Pesaguero",
			'39051' => "Pesquera",
			'39052' => "Piélagos",
			'39053' => "Polaciones",
			'39054' => "Polanco",
			'39055' => "Potes",
			'39056' => "Puente Viesgo",
			'39057' => "Ramales de la Victoria",
			'39058' => "Rasines",
			'39059' => "Reinosa",
			'39060' => "Reocín",
			'39061' => "Ribamontán al Mar",
			'39062' => "Ribamontán al Monte",
			'39063' => "Rionansa",
			'39064' => "Riotuerto",
			'39065' => "Rozas de Valdearroyo, Las",
			'39066' => "Ruente",
			'39067' => "Ruesga",
			'39068' => "Ruiloba",
			'39069' => "San Felices de Buelna",
			'39070' => "San Miguel de Aguayo",
			'39071' => "San Pedro del Romeral",
			'39072' => "San Roque de Riomiera",
			'39080' => "San Vicente de la Barquera",
			'39073' => "Santa Cruz de Bezana",
			'39074' => "Santa María de Cayón",
			'39075' => "Santander",
			'39076' => "Santillana del Mar",
			'39077' => "Santiurde de Reinosa",
			'39078' => "Santiurde de Toranzo",
			'39079' => "Santoña",
			'39081' => "Saro",
			'39082' => "Selaya",
			'39083' => "Soba",
			'39084' => "Solórzano",
			'39085' => "Suances",
			'39086' => "Tojos, Los",
			'39087' => "Torrelavega",
			'39088' => "Tresviso",
			'39089' => "Tudanca",
			'39090' => "Udías",
			'39095' => "Val de San Vicente",
			'39091' => "Valdáliga",
			'39092' => "Valdeolea",
			'39093' => "Valdeprado del Río",
			'39094' => "Valderredible",
			'39101' => "Valle de Villaverde",
			'39096' => "Vega de Liébana",
			'39097' => "Vega de Pas",
			'39098' => "Villacarriedo",
			'39099' => "Villaescusa",
			'39100' => "Villafufre",
			'39102' => "Voto",
			'40001' => "Abades",
			'40002' => "Adrada de Pirón",
			'40003' => "Adrados",
			'40004' => "Aguilafuente",
			'40005' => "Alconada de Maderuelo",
			'40012' => "Aldea Real",
			'40006' => "Aldealcorvo",
			'40007' => "Aldealengua de Pedraza",
			'40008' => "Aldealengua de Santa María",
			'40009' => "Aldeanueva de la Serrezuela",
			'40010' => "Aldeanueva del Codonal",
			'40013' => "Aldeasoña",
			'40014' => "Aldehorno",
			'40015' => "Aldehuela del Codonal",
			'40016' => "Aldeonte",
			'40017' => "Anaya",
			'40018' => "Añe",
			'40019' => "Arahuetes",
			'40020' => "Arcones",
			'40021' => "Arevalillo de Cega",
			'40022' => "Armuña",
			'40024' => "Ayllón",
			'40025' => "Barbolla",
			'40026' => "Basardilla",
			'40028' => "Bercial",
			'40029' => "Bercimuel",
			'40030' => "Bernardos",
			'40031' => "Bernuy de Porreros",
			'40032' => "Boceguillas",
			'40033' => "Brieva",
			'40034' => "Caballar",
			'40035' => "Cabañas de Polendos",
			'40036' => "Cabezuela",
			'40037' => "Calabazas de Fuentidueña",
			'40039' => "Campo de San Pedro",
			'40040' => "Cantalejo",
			'40041' => "Cantimpalos",
			'40161' => "Carabias",
			'40043' => "Carbonero el Mayor",
			'40044' => "Carrascal del Río",
			'40045' => "Casla",
			'40046' => "Castillejo de Mesleón",
			'40047' => "Castro de Fuentidueña",
			'40048' => "Castrojimeno",
			'40049' => "Castroserna de Abajo",
			'40051' => "Castroserracín",
			'40052' => "Cedillo de la Torre",
			'40053' => "Cerezo de Abajo",
			'40054' => "Cerezo de Arriba",
			'40065' => "Chañe",
			'40055' => "Cilleruelo de San Mamés",
			'40056' => "Cobos de Fuentidueña",
			'40057' => "Coca",
			'40058' => "Codorniz",
			'40059' => "Collado Hermoso",
			'40060' => "Condado de Castilnovo",
			'40061' => "Corral de Ayllón",
			'40902' => "Cozuelos de Fuentidueña",
			'40062' => "Cubillo",
			'40063' => "Cuéllar",
			'40905' => "Cuevas de Provanco",
			'40068' => "Domingo García",
			'40069' => "Donhierro",
			'40070' => "Duruelo",
			'40071' => "Encinas",
			'40072' => "Encinillas",
			'40073' => "Escalona del Prado",
			'40074' => "Escarabajosa de Cabezas",
			'40075' => "Escobar de Polendos",
			'40076' => "Espinar, El",
			'40077' => "Espirdo",
			'40078' => "Fresneda de Cuéllar",
			'40079' => "Fresno de Cantespino",
			'40080' => "Fresno de la Fuente",
			'40081' => "Frumales",
			'40082' => "Fuente de Santa Cruz",
			'40083' => "Fuente el Olmo de Fuentidueña",
			'40084' => "Fuente el Olmo de Íscar",
			'40086' => "Fuentepelayo",
			'40087' => "Fuentepiñel",
			'40088' => "Fuenterrebollo",
			'40089' => "Fuentesaúco de Fuentidueña",
			'40091' => "Fuentesoto",
			'40092' => "Fuentidueña",
			'40093' => "Gallegos",
			'40094' => "Garcillán",
			'40095' => "Gomezserracín",
			'40097' => "Grajera",
			'40099' => "Honrubia de la Cuesta",
			'40100' => "Hontalbilla",
			'40101' => "Hontanares de Eresma",
			'40103' => "Huertos, Los",
			'40104' => "Ituero y Lama",
			'40105' => "Juarros de Riomoros",
			'40106' => "Juarros de Voltoya",
			'40107' => "Labajos",
			'40108' => "Laguna de Contreras",
			'40109' => "Languilla",
			'40110' => "Lastras de Cuéllar",
			'40111' => "Lastras del Pozo",
			'40112' => "Lastrilla, La",
			'40113' => "Losa, La",
			'40115' => "Maderuelo",
			'40903' => "Marazoleja",
			'40118' => "Marazuela",
			'40119' => "Martín Miguel",
			'40120' => "Martín Muñoz de la Dehesa",
			'40121' => "Martín Muñoz de las Posadas",
			'40122' => "Marugán",
			'40124' => "Mata de Cuéllar",
			'40123' => "Matabuena",
			'40125' => "Matilla, La",
			'40126' => "Melque de Cercos",
			'40127' => "Membibre de la Hoz",
			'40128' => "Migueláñez",
			'40129' => "Montejo de Arévalo",
			'40130' => "Montejo de la Vega de la Serrezuela",
			'40131' => "Monterrubio",
			'40132' => "Moral de Hornuez",
			'40134' => "Mozoncillo",
			'40135' => "Muñopedro",
			'40136' => "Muñoveros",
			'40138' => "Nava de la Asunción",
			'40139' => "Navafría",
			'40140' => "Navalilla",
			'40141' => "Navalmanzano",
			'40142' => "Navares de Ayuso",
			'40143' => "Navares de Enmedio",
			'40144' => "Navares de las Cuevas",
			'40145' => "Navas de Oro",
			'40904' => "Navas de Riofrío",
			'40146' => "Navas de San Antonio",
			'40148' => "Nieva",
			'40149' => "Olombrada",
			'40150' => "Orejana",
			'40151' => "Ortigosa de Pestaño",
			'40901' => "Ortigosa del Monte",
			'40152' => "Otero de Herreros",
			'40154' => "Pajarejos",
			'40155' => "Palazuelos de Eresma",
			'40156' => "Pedraza",
			'40157' => "Pelayos del Arroyo",
			'40158' => "Perosillo",
			'40159' => "Pinarejos",
			'40160' => "Pinarnegrillo",
			'40162' => "Prádena",
			'40163' => "Puebla de Pedraza",
			'40164' => "Rapariegos",
			'40181' => "Real Sitio de San Ildefonso",
			'40165' => "Rebollo",
			'40166' => "Remondo",
			'40168' => "Riaguas de San Bartolomé",
			'40170' => "Riaza",
			'40171' => "Ribota",
			'40172' => "Riofrío de Riaza",
			'40173' => "Roda de Eresma",
			'40174' => "Sacramenia",
			'40176' => "Samboal",
			'40177' => "San Cristóbal de Cuéllar",
			'40178' => "San Cristóbal de la Vega",
			'40906' => "San Cristóbal de Segovia",
			'40182' => "San Martín y Mudrián",
			'40183' => "San Miguel de Bernuy",
			'40184' => "San Pedro de Gaíllos",
			'40179' => "Sanchonuño",
			'40180' => "Sangarcía",
			'40185' => "Santa María la Real de Nieva",
			'40186' => "Santa Marta del Cerro",
			'40188' => "Santiuste de Pedraza",
			'40189' => "Santiuste de San Juan Bautista",
			'40190' => "Santo Domingo de Pirón",
			'40191' => "Santo Tomé del Puerto",
			'40192' => "Sauquillo de Cabezas",
			'40193' => "Sebúlcor",
			'40194' => "Segovia",
			'40195' => "Sepúlveda",
			'40196' => "Sequera de Fresno",
			'40198' => "Sotillo",
			'40199' => "Sotosalbos",
			'40200' => "Tabanera la Luenga",
			'40201' => "Tolocirio",
			'40206' => "Torre Val de San Pedro",
			'40202' => "Torreadrada",
			'40203' => "Torrecaballeros",
			'40204' => "Torrecilla del Pinar",
			'40205' => "Torreiglesias",
			'40207' => "Trescasas",
			'40208' => "Turégano",
			'40210' => "Urueñas",
			'40211' => "Valdeprados",
			'40212' => "Valdevacas de Montejo",
			'40213' => "Valdevacas y Guijar",
			'40218' => "Valle de Tabladillo",
			'40219' => "Vallelado",
			'40220' => "Valleruela de Pedraza",
			'40221' => "Valleruela de Sepúlveda",
			'40214' => "Valseca",
			'40215' => "Valtiendas",
			'40216' => "Valverde del Majano",
			'40222' => "Veganzones",
			'40223' => "Vegas de Matute",
			'40224' => "Ventosilla y Tejadilla",
			'40225' => "Villacastín",
			'40228' => "Villaverde de Íscar",
			'40229' => "Villaverde de Montejo",
			'40230' => "Villeguillo",
			'40231' => "Yanguas de Eresma",
			'40233' => "Zarzuela del Monte",
			'40234' => "Zarzuela del Pinar",
			'41001' => "Aguadulce",
			'41002' => "Alanís",
			'41003' => "Albaida del Aljarafe",
			'41004' => "Alcalá de Guadaíra",
			'41005' => "Alcalá del Río",
			'41006' => "Alcolea del Río",
			'41007' => "Algaba, La",
			'41008' => "Algámitas",
			'41009' => "Almadén de la Plata",
			'41010' => "Almensilla",
			'41011' => "Arahal",
			'41012' => "Aznalcázar",
			'41013' => "Aznalcóllar",
			'41014' => "Badolatosa",
			'41015' => "Benacazón",
			'41016' => "Bollullos de la Mitación",
			'41017' => "Bormujos",
			'41018' => "Brenes",
			'41019' => "Burguillos",
			'41020' => "Cabezas de San Juan, Las",
			'41021' => "Camas",
			'41022' => "Campana, La",
			'41023' => "Cantillana",
			'41901' => "Cañada Rosal",
			'41024' => "Carmona",
			'41025' => "Carrión de los Céspedes",
			'41026' => "Casariche",
			'41027' => "Castilblanco de los Arroyos",
			'41028' => "Castilleja de Guzmán",
			'41029' => "Castilleja de la Cuesta",
			'41030' => "Castilleja del Campo",
			'41031' => "Castillo de las Guardas, El",
			'41032' => "Cazalla de la Sierra",
			'41033' => "Constantina",
			'41034' => "Coria del Río",
			'41035' => "Coripe",
			'41036' => "Coronil, El",
			'41037' => "Corrales, Los",
			'41903' => "Cuervo de Sevilla, El",
			'41038' => "Dos Hermanas",
			'41039' => "Écija",
			'41040' => "Espartinas",
			'41041' => "Estepa",
			'41042' => "Fuentes de Andalucía",
			'41043' => "Garrobo, El",
			'41044' => "Gelves",
			'41045' => "Gerena",
			'41046' => "Gilena",
			'41047' => "Gines",
			'41048' => "Guadalcanal",
			'41049' => "Guillena",
			'41050' => "Herrera",
			'41051' => "Huévar del Aljarafe",
			'41902' => "Isla Mayor",
			'41052' => "Lantejuela",
			'41053' => "Lebrija",
			'41054' => "Lora de Estepa",
			'41055' => "Lora del Río",
			'41056' => "Luisiana, La",
			'41057' => "Madroño, El",
			'41058' => "Mairena del Alcor",
			'41059' => "Mairena del Aljarafe",
			'41060' => "Marchena",
			'41061' => "Marinaleda",
			'41062' => "Martín de la Jara",
			'41063' => "Molares, Los",
			'41064' => "Montellano",
			'41065' => "Morón de la Frontera",
			'41066' => "Navas de la Concepción, Las",
			'41067' => "Olivares",
			'41068' => "Osuna",
			'41069' => "Palacios y Villafranca, Los",
			'41904' => "Palmar de Troya, El",
			'41070' => "Palomares del Río",
			'41071' => "Paradas",
			'41072' => "Pedrera",
			'41073' => "Pedroso, El",
			'41074' => "Peñaflor",
			'41075' => "Pilas",
			'41076' => "Pruna",
			'41077' => "Puebla de Cazalla, La",
			'41078' => "Puebla de los Infantes, La",
			'41079' => "Puebla del Río, La",
			'41080' => "Real de la Jara, El",
			'41081' => "Rinconada, La",
			'41082' => "Roda de Andalucía, La",
			'41083' => "Ronquillo, El",
			'41084' => "Rubio, El",
			'41085' => "Salteras",
			'41086' => "San Juan de Aznalfarache",
			'41088' => "San Nicolás del Puerto",
			'41087' => "Sanlúcar la Mayor",
			'41089' => "Santiponce",
			'41090' => "Saucejo, El",
			'41091' => "Sevilla",
			'41092' => "Tocina",
			'41093' => "Tomares",
			'41094' => "Umbrete",
			'41095' => "Utrera",
			'41096' => "Valencina de la Concepción",
			'41097' => "Villamanrique de la Condesa",
			'41100' => "Villanueva de San Juan",
			'41098' => "Villanueva del Ariscal",
			'41099' => "Villanueva del Río y Minas",
			'41101' => "Villaverde del Río",
			'41102' => "Viso del Alcor, El",
			'42001' => "Abejar",
			'42003' => "Adradas",
			'42004' => "Ágreda",
			'42006' => "Alconaba",
			'42007' => "Alcubilla de Avellaneda",
			'42008' => "Alcubilla de las Peñas",
			'42009' => "Aldealafuente",
			'42010' => "Aldealices",
			'42011' => "Aldealpozo",
			'42012' => "Aldealseñor",
			'42013' => "Aldehuela de Periáñez",
			'42014' => "Aldehuelas, Las",
			'42015' => "Alentisque",
			'42016' => "Aliud",
			'42017' => "Almajano",
			'42018' => "Almaluez",
			'42019' => "Almarza",
			'42020' => "Almazán",
			'42021' => "Almazul",
			'42022' => "Almenar de Soria",
			'42023' => "Alpanseque",
			'42024' => "Arancón",
			'42025' => "Arcos de Jalón",
			'42026' => "Arenillas",
			'42027' => "Arévalo de la Sierra",
			'42028' => "Ausejo de la Sierra",
			'42029' => "Baraona",
			'42030' => "Barca",
			'42031' => "Barcones",
			'42032' => "Bayubas de Abajo",
			'42033' => "Bayubas de Arriba",
			'42034' => "Beratón",
			'42035' => "Berlanga de Duero",
			'42036' => "Blacos",
			'42037' => "Bliecos",
			'42038' => "Borjabad",
			'42039' => "Borobia",
			'42041' => "Buberos",
			'42042' => "Buitrago",
			'42043' => "Burgo de Osma-Ciudad de Osma",
			'42044' => "Cabrejas del Campo",
			'42045' => "Cabrejas del Pinar",
			'42046' => "Calatañazor",
			'42048' => "Caltojar",
			'42049' => "Candilichera",
			'42050' => "Cañamaque",
			'42051' => "Carabantes",
			'42052' => "Caracena",
			'42053' => "Carrascosa de Abajo",
			'42054' => "Carrascosa de la Sierra",
			'42055' => "Casarejos",
			'42056' => "Castilfrío de la Sierra",
			'42058' => "Castillejo de Robledo",
			'42057' => "Castilruiz",
			'42059' => "Centenera de Andaluz",
			'42060' => "Cerbón",
			'42061' => "Cidones",
			'42062' => "Cigudosa",
			'42063' => "Cihuela",
			'42064' => "Ciria",
			'42065' => "Cirujales del Río",
			'42068' => "Coscurita",
			'42069' => "Covaleda",
			'42070' => "Cubilla",
			'42071' => "Cubo de la Solana",
			'42073' => "Cueva de Ágreda",
			'42075' => "Dévanos",
			'42076' => "Deza",
			'42078' => "Duruelo de la Sierra",
			'42079' => "Escobosa de Almazán",
			'42080' => "Espeja de San Marcelino",
			'42081' => "Espejón",
			'42082' => "Estepa de San Juan",
			'42083' => "Frechilla de Almazán",
			'42084' => "Fresno de Caracena",
			'42085' => "Fuentearmegil",
			'42086' => "Fuentecambrón",
			'42087' => "Fuentecantos",
			'42088' => "Fuentelmonge",
			'42089' => "Fuentelsaz de Soria",
			'42090' => "Fuentepinilla",
			'42092' => "Fuentes de Magaña",
			'42093' => "Fuentestrún",
			'42094' => "Garray",
			'42095' => "Golmayo",
			'42096' => "Gómara",
			'42097' => "Gormaz",
			'42098' => "Herrera de Soria",
			'42100' => "Hinojosa del Campo",
			'42103' => "Langa de Duero",
			'42105' => "Liceras",
			'42106' => "Losilla, La",
			'42107' => "Magaña",
			'42108' => "Maján",
			'42110' => "Matalebreras",
			'42111' => "Matamala de Almazán",
			'42113' => "Medinaceli",
			'42115' => "Miño de Medinaceli",
			'42116' => "Miño de San Esteban",
			'42117' => "Molinos de Duero",
			'42118' => "Momblona",
			'42119' => "Monteagudo de las Vicarías",
			'42120' => "Montejo de Tiermes",
			'42121' => "Montenegro de Cameros",
			'42123' => "Morón de Almazán",
			'42124' => "Muriel de la Fuente",
			'42125' => "Muriel Viejo",
			'42127' => "Nafría de Ucero",
			'42128' => "Narros",
			'42129' => "Navaleno",
			'42130' => "Nepas",
			'42131' => "Nolay",
			'42132' => "Noviercas",
			'42134' => "Ólvega",
			'42135' => "Oncala",
			'42139' => "Pinilla del Campo",
			'42140' => "Portillo de Soria",
			'42141' => "Póveda de Soria, La",
			'42142' => "Pozalmuro",
			'42144' => "Quintana Redonda",
			'42145' => "Quintanas de Gormaz",
			'42148' => "Quiñonería",
			'42149' => "Rábanos, Los",
			'42151' => "Rebollar",
			'42152' => "Recuerda",
			'42153' => "Rello",
			'42154' => "Renieblas",
			'42155' => "Retortillo de Soria",
			'42156' => "Reznos",
			'42157' => "Riba de Escalote, La",
			'42158' => "Rioseco de Soria",
			'42159' => "Rollamienta",
			'42160' => "Royo, El",
			'42161' => "Salduero",
			'42162' => "San Esteban de Gormaz",
			'42163' => "San Felices",
			'42164' => "San Leonardo de Yagüe",
			'42165' => "San Pedro Manrique",
			'42166' => "Santa Cruz de Yanguas",
			'42167' => "Santa María de Huerta",
			'42168' => "Santa María de las Hoyas",
			'42171' => "Serón de Nágima",
			'42172' => "Soliedra",
			'42173' => "Soria",
			'42174' => "Sotillo del Rincón",
			'42175' => "Suellacabras",
			'42176' => "Tajahuerce",
			'42177' => "Tajueco",
			'42178' => "Talveila",
			'42181' => "Tardelcuende",
			'42182' => "Taroda",
			'42183' => "Tejado",
			'42184' => "Torlengua",
			'42185' => "Torreblacos",
			'42187' => "Torrubia de Soria",
			'42188' => "Trévago",
			'42189' => "Ucero",
			'42190' => "Vadillo",
			'42191' => "Valdeavellano de Tera",
			'42192' => "Valdegeña",
			'42193' => "Valdelagua del Cerro",
			'42194' => "Valdemaluque",
			'42195' => "Valdenebro",
			'42196' => "Valdeprado",
			'42197' => "Valderrodilla",
			'42198' => "Valtajeros",
			'42200' => "Velamazán",
			'42201' => "Velilla de la Sierra",
			'42202' => "Velilla de los Ajos",
			'42204' => "Viana de Duero",
			'42205' => "Villaciervos",
			'42206' => "Villanueva de Gormaz",
			'42207' => "Villar del Ala",
			'42208' => "Villar del Campo",
			'42209' => "Villar del Río",
			'42211' => "Villares de Soria, Los",
			'42212' => "Villasayas",
			'42213' => "Villaseca de Arciel",
			'42215' => "Vinuesa",
			'42216' => "Vizmanos",
			'42217' => "Vozmediano",
			'42218' => "Yanguas",
			'42219' => "Yelo",
			'43001' => "Aiguamúrcia",
			'43002' => "Albinyana",
			'43003' => "Albiol, L'",
			'43004' => "Alcanar",
			'43005' => "Alcover",
			'43904' => "Aldea, L'",
			'43006' => "Aldover",
			'43007' => "Aleixar, L'",
			'43008' => "Alfara de Carles",
			'43009' => "Alforja",
			'43010' => "Alió",
			'43011' => "Almoster",
			'43012' => "Altafulla",
			'43013' => "Ametlla de Mar, L'",
			'43906' => "Ampolla, L'",
			'43014' => "Amposta",
			'43016' => "Arboç, L'",
			'43015' => "Arbolí",
			'43017' => "Argentera, L'",
			'43018' => "Arnes",
			'43019' => "Ascó",
			'43020' => "Banyeres del Penedès",
			'43021' => "Barberà de la Conca",
			'43022' => "Batea",
			'43023' => "Bellmunt del Priorat",
			'43024' => "Bellvei",
			'43025' => "Benifallet",
			'43026' => "Benissanet",
			'43027' => "Bisbal de Falset, La",
			'43028' => "Bisbal del Penedès, La",
			'43029' => "Blancafort",
			'43030' => "Bonastre",
			'43031' => "Borges del Camp, Les",
			'43032' => "Bot",
			'43033' => "Botarell",
			'43034' => "Bràfim",
			'43035' => "Cabacés",
			'43036' => "Cabra del Camp",
			'43037' => "Calafell",
			'43903' => "Camarles",
			'43038' => "Cambrils",
			'43907' => "Canonja, La",
			'43039' => "Capafonts",
			'43040' => "Capçanes",
			'43041' => "Caseres",
			'43042' => "Castellvell del Camp",
			'43043' => "Catllar, El",
			'43045' => "Colldejou",
			'43046' => "Conesa",
			'43047' => "Constantí",
			'43048' => "Corbera d'Ebre",
			'43049' => "Cornudella de Montsant",
			'43050' => "Creixell",
			'43051' => "Cunit",
			'43901' => "Deltebre",
			'43053' => "Duesaigües",
			'43054' => "Espluga de Francolí, L'",
			'43055' => "Falset",
			'43056' => "Fatarella, La",
			'43057' => "Febró, La",
			'43058' => "Figuera, La",
			'43059' => "Figuerola del Camp",
			'43060' => "Flix",
			'43061' => "Forès",
			'43062' => "Freginals",
			'43063' => "Galera, La",
			'43064' => "Gandesa",
			'43065' => "Garcia",
			'43066' => "Garidells, Els",
			'43067' => "Ginestar",
			'43068' => "Godall",
			'43069' => "Gratallops",
			'43070' => "Guiamets, Els",
			'43071' => "Horta de Sant Joan",
			'43072' => "Lloar, El",
			'43073' => "Llorac",
			'43074' => "Llorenç del Penedès",
			'43076' => "Marçà",
			'43075' => "Margalef",
			'43077' => "Mas de Barberans",
			'43078' => "Masdenverge",
			'43079' => "Masllorenç",
			'43080' => "Masó, La",
			'43081' => "Maspujols",
			'43082' => "Masroig, El",
			'43083' => "Milà, El",
			'43084' => "Miravet",
			'43085' => "Molar, El",
			'43086' => "Montblanc",
			'43088' => "Montbrió del Camp",
			'43089' => "Montferri",
			'43090' => "Montmell, El",
			'43091' => "Mont-ral",
			'43092' => "Mont-roig del Camp",
			'43093' => "Móra d'Ebre",
			'43094' => "Móra la Nova",
			'43095' => "Morell, El",
			'43096' => "Morera de Montsant, La",
			'43097' => "Nou de Gaià, La",
			'43098' => "Nulles",
			'43100' => "Pallaresos, Els",
			'43099' => "Palma d'Ebre, La",
			'43101' => "Passanant i Belltall",
			'43102' => "Paüls",
			'43103' => "Perafort",
			'43104' => "Perelló, El",
			'43105' => "Piles, Les",
			'43106' => "Pinell de Brai, El",
			'43107' => "Pira",
			'43108' => "Pla de Santa Maria, El",
			'43109' => "Pobla de Mafumet, La",
			'43110' => "Pobla de Massaluca, La",
			'43111' => "Pobla de Montornès, La",
			'43112' => "Poboleda",
			'43113' => "Pont d'Armentera, El",
			'43141' => "Pontils",
			'43114' => "Porrera",
			'43115' => "Pradell de la Teixeta",
			'43116' => "Prades",
			'43117' => "Prat de Comte",
			'43118' => "Pratdip",
			'43119' => "Puigpelat",
			'43120' => "Querol",
			'43136' => "Ràpita, La",
			'43121' => "Rasquera",
			'43122' => "Renau",
			'43123' => "Reus",
			'43124' => "Riba, La",
			'43125' => "Riba-roja d'Ebre",
			'43126' => "Riera de Gaià, La",
			'43127' => "Riudecanyes",
			'43128' => "Riudecols",
			'43129' => "Riudoms",
			'43130' => "Rocafort de Queralt",
			'43131' => "Roda de Berà",
			'43132' => "Rodonyà",
			'43133' => "Roquetes",
			'43134' => "Rourell, El",
			'43135' => "Salomó",
			'43905' => "Salou",
			'43137' => "Sant Jaume dels Domenys",
			'43902' => "Sant Jaume d'Enveja",
			'43138' => "Santa Bàrbara",
			'43139' => "Santa Coloma de Queralt",
			'43140' => "Santa Oliva",
			'43142' => "Sarral",
			'43143' => "Savallà del Comtat",
			'43144' => "Secuita, La",
			'43145' => "Selva del Camp, La",
			'43146' => "Senan",
			'43044' => "Sénia, La",
			'43147' => "Solivella",
			'43148' => "Tarragona",
			'43149' => "Tivenys",
			'43150' => "Tivissa",
			'43151' => "Torre de Fontaubella, La",
			'43152' => "Torre de l'Espanyol, La",
			'43153' => "Torredembarra",
			'43154' => "Torroja del Priorat",
			'43155' => "Tortosa",
			'43156' => "Ulldecona",
			'43157' => "Ulldemolins",
			'43158' => "Vallclara",
			'43159' => "Vallfogona de Riucorb",
			'43160' => "Vallmoll",
			'43161' => "Valls",
			'43162' => "Vandellòs i l'Hospitalet de l'Infant",
			'43163' => "Vendrell, El",
			'43164' => "Vespella de Gaià",
			'43165' => "Vilabella",
			'43175' => "Vilalba dels Arcs",
			'43166' => "Vilallonga del Camp",
			'43168' => "Vilanova de Prades",
			'43167' => "Vilanova d'Escornalbou",
			'43169' => "Vilaplana",
			'43170' => "Vila-rodona",
			'43171' => "Vila-seca",
			'43172' => "Vilaverd",
			'43173' => "Vilella Alta, La",
			'43174' => "Vilella Baixa, La",
			'43176' => "Vimbodí i Poblet",
			'43177' => "Vinebre",
			'43178' => "Vinyols i els Arcs",
			'43052' => "Xerta",
			'44001' => "Ababuj",
			'44002' => "Abejuela",
			'44003' => "Aguatón",
			'44004' => "Aguaviva",
			'44005' => "Aguilar del Alfambra",
			'44006' => "Alacón",
			'44007' => "Alba",
			'44008' => "Albalate del Arzobispo",
			'44009' => "Albarracín",
			'44010' => "Albentosa",
			'44011' => "Alcaine",
			'44012' => "Alcalá de la Selva",
			'44013' => "Alcañiz",
			'44014' => "Alcorisa",
			'44016' => "Alfambra",
			'44017' => "Aliaga",
			'44021' => "Allepuz",
			'44022' => "Alloza",
			'44023' => "Allueva",
			'44018' => "Almohaja",
			'44019' => "Alobras",
			'44020' => "Alpeñés",
			'44024' => "Anadón",
			'44025' => "Andorra",
			'44026' => "Arcos de las Salinas",
			'44027' => "Arens de Lledó",
			'44028' => "Argente",
			'44029' => "Ariño",
			'44031' => "Azaila",
			'44032' => "Bádenas",
			'44033' => "Báguena",
			'44034' => "Bañón",
			'44035' => "Barrachina",
			'44036' => "Bea",
			'44037' => "Beceite",
			'44039' => "Bello",
			'44038' => "Belmonte de San José",
			'44040' => "Berge",
			'44041' => "Bezas",
			'44042' => "Blancas",
			'44043' => "Blesa",
			'44044' => "Bordón",
			'44045' => "Bronchales",
			'44046' => "Bueña",
			'44047' => "Burbáguena",
			'44048' => "Cabra de Mora",
			'44049' => "Calaceite",
			'44050' => "Calamocha",
			'44051' => "Calanda",
			'44052' => "Calomarde",
			'44053' => "Camañas",
			'44054' => "Camarena de la Sierra",
			'44055' => "Camarillas",
			'44056' => "Caminreal",
			'44059' => "Cantavieja",
			'44060' => "Cañada de Benatanduz",
			'44061' => "Cañada de Verich, La",
			'44062' => "Cañada Vellida",
			'44063' => "Cañizar del Olivar",
			'44064' => "Cascante del Río",
			'44065' => "Castejón de Tornos",
			'44066' => "Castel de Cabra",
			'44070' => "Castellar, El",
			'44071' => "Castellote",
			'44067' => "Castelnou",
			'44068' => "Castelserás",
			'44074' => "Cedrillas",
			'44075' => "Celadas",
			'44076' => "Cella",
			'44077' => "Cerollera, La",
			'44080' => "Codoñera, La",
			'44082' => "Corbalán",
			'44084' => "Cortes de Aragón",
			'44085' => "Cosa",
			'44086' => "Cretas",
			'44087' => "Crivillén",
			'44088' => "Cuba, La",
			'44089' => "Cubla",
			'44090' => "Cucalón",
			'44092' => "Cuervo, El",
			'44093' => "Cuevas de Almudén",
			'44094' => "Cuevas Labradas",
			'44096' => "Ejulve",
			'44097' => "Escorihuela",
			'44099' => "Escucha",
			'44100' => "Estercuel",
			'44101' => "Ferreruela de Huerva",
			'44102' => "Fonfría",
			'44103' => "Formiche Alto",
			'44105' => "Fórnoles",
			'44106' => "Fortanete",
			'44107' => "Foz-Calanda",
			'44108' => "Fresneda, La",
			'44109' => "Frías de Albarracín",
			'44110' => "Fuenferrada",
			'44111' => "Fuentes Calientes",
			'44112' => "Fuentes Claras",
			'44113' => "Fuentes de Rubielos",
			'44114' => "Fuentespalda",
			'44115' => "Galve",
			'44116' => "Gargallo",
			'44117' => "Gea de Albarracín",
			'44118' => "Ginebrosa, La",
			'44119' => "Griegos",
			'44120' => "Guadalaviar",
			'44121' => "Gúdar",
			'44122' => "Híjar",
			'44123' => "Hinojosa de Jarque",
			'44124' => "Hoz de la Vieja, La",
			'44125' => "Huesa del Común",
			'44126' => "Iglesuela del Cid, La",
			'44127' => "Jabaloyas",
			'44128' => "Jarque de la Val",
			'44129' => "Jatiel",
			'44130' => "Jorcas",
			'44131' => "Josa",
			'44132' => "Lagueruela",
			'44133' => "Lanzuela",
			'44135' => "Libros",
			'44136' => "Lidón",
			'44137' => "Linares de Mora",
			'44141' => "Lledó",
			'44138' => "Loscos",
			'44142' => "Maicas",
			'44143' => "Manzanera",
			'44144' => "Martín del Río",
			'44145' => "Mas de las Matas",
			'44146' => "Mata de los Olmos, La",
			'44147' => "Mazaleón",
			'44148' => "Mezquita de Jarque",
			'44149' => "Mirambel",
			'44150' => "Miravete de la Sierra",
			'44151' => "Molinos",
			'44152' => "Monforte de Moyuela",
			'44153' => "Monreal del Campo",
			'44154' => "Monroyo",
			'44155' => "Montalbán",
			'44156' => "Monteagudo del Castillo",
			'44157' => "Monterde de Albarracín",
			'44158' => "Mora de Rubielos",
			'44159' => "Moscardón",
			'44160' => "Mosqueruela",
			'44161' => "Muniesa",
			'44163' => "Noguera de Albarracín",
			'44164' => "Nogueras",
			'44165' => "Nogueruelas",
			'44167' => "Obón",
			'44168' => "Odón",
			'44169' => "Ojos Negros",
			'44171' => "Olba",
			'44172' => "Oliete",
			'44173' => "Olmos, Los",
			'44174' => "Orihuela del Tremedal",
			'44175' => "Orrios",
			'44176' => "Palomar de Arroyos",
			'44177' => "Pancrudo",
			'44178' => "Parras de Castellote, Las",
			'44179' => "Peñarroya de Tastavins",
			'44180' => "Peracense",
			'44181' => "Peralejos",
			'44182' => "Perales del Alfambra",
			'44183' => "Pitarque",
			'44184' => "Plou",
			'44185' => "Pobo, El",
			'44187' => "Portellada, La",
			'44189' => "Pozondón",
			'44190' => "Pozuel del Campo",
			'44191' => "Puebla de Híjar, La",
			'44192' => "Puebla de Valverde, La",
			'44193' => "Puertomingalvo",
			'44194' => "Ráfales",
			'44195' => "Rillo",
			'44196' => "Riodeva",
			'44197' => "Ródenas",
			'44198' => "Royuela",
			'44199' => "Rubiales",
			'44200' => "Rubielos de la Cérida",
			'44201' => "Rubielos de Mora",
			'44203' => "Salcedillo",
			'44204' => "Saldón",
			'44205' => "Samper de Calanda",
			'44206' => "San Agustín",
			'44207' => "San Martín del Río",
			'44208' => "Santa Cruz de Nogueras",
			'44209' => "Santa Eulalia",
			'44210' => "Sarrión",
			'44211' => "Segura de los Baños",
			'44212' => "Seno",
			'44213' => "Singra",
			'44215' => "Terriente",
			'44216' => "Teruel",
			'44217' => "Toril y Masegoso",
			'44218' => "Tormón",
			'44219' => "Tornos",
			'44220' => "Torralba de los Sisones",
			'44223' => "Torre de Arcas",
			'44224' => "Torre de las Arcas",
			'44225' => "Torre del Compte",
			'44227' => "Torre los Negros",
			'44221' => "Torrecilla de Alcañiz",
			'44222' => "Torrecilla del Rebollar",
			'44226' => "Torrelacárcel",
			'44228' => "Torremocha de Jiloca",
			'44229' => "Torres de Albarracín",
			'44230' => "Torrevelilla",
			'44231' => "Torrijas",
			'44232' => "Torrijo del Campo",
			'44234' => "Tramacastiel",
			'44235' => "Tramacastilla",
			'44236' => "Tronchón",
			'44237' => "Urrea de Gaén",
			'44238' => "Utrillas",
			'44239' => "Valacloche",
			'44240' => "Valbona",
			'44241' => "Valdealgorfa",
			'44243' => "Valdecuenca",
			'44244' => "Valdelinares",
			'44245' => "Valdeltormo",
			'44246' => "Valderrobres",
			'44247' => "Valjunquera",
			'44249' => "Vallecillo, El",
			'44250' => "Veguillas de la Sierra",
			'44251' => "Villafranca del Campo",
			'44252' => "Villahermosa del Campo",
			'44256' => "Villanueva del Rebollar de la Sierra",
			'44257' => "Villar del Cobo",
			'44258' => "Villar del Salz",
			'44260' => "Villarluengo",
			'44261' => "Villarquemado",
			'44262' => "Villarroya de los Pinares",
			'44263' => "Villastar",
			'44264' => "Villel",
			'44265' => "Vinaceite",
			'44266' => "Visiedo",
			'44267' => "Vivel del Río Martín",
			'44268' => "Zoma, La",
			'45001' => "Ajofrín",
			'45002' => "Alameda de la Sagra",
			'45003' => "Albarreal de Tajo",
			'45004' => "Alcabón",
			'45005' => "Alcañizo",
			'45006' => "Alcaudete de la Jara",
			'45007' => "Alcolea de Tajo",
			'45008' => "Aldea en Cabo",
			'45009' => "Aldeanueva de Barbarroya",
			'45010' => "Aldeanueva de San Bartolomé",
			'45011' => "Almendral de la Cañada",
			'45012' => "Almonacid de Toledo",
			'45013' => "Almorox",
			'45014' => "Añover de Tajo",
			'45015' => "Arcicóllar",
			'45016' => "Argés",
			'45017' => "Azután",
			'45018' => "Barcience",
			'45019' => "Bargas",
			'45020' => "Belvís de la Jara",
			'45021' => "Borox",
			'45022' => "Buenaventura",
			'45023' => "Burguillos de Toledo",
			'45024' => "Burujón",
			'45025' => "Cabañas de la Sagra",
			'45026' => "Cabañas de Yepes",
			'45027' => "Cabezamesada",
			'45028' => "Calera y Chozas",
			'45029' => "Caleruela",
			'45030' => "Calzada de Oropesa",
			'45031' => "Camarena",
			'45032' => "Camarenilla",
			'45033' => "Campillo de la Jara, El",
			'45034' => "Camuñas",
			'45035' => "Cardiel de los Montes",
			'45036' => "Carmena",
			'45037' => "Carpio de Tajo, El",
			'45038' => "Carranque",
			'45039' => "Carriches",
			'45040' => "Casar de Escalona, El",
			'45041' => "Casarrubios del Monte",
			'45042' => "Casasbuenas",
			'45043' => "Castillo de Bayuela",
			'45045' => "Cazalegas",
			'45046' => "Cebolla",
			'45047' => "Cedillo del Condado",
			'45048' => "Cerralbos, Los",
			'45049' => "Cervera de los Montes",
			'45056' => "Chozas de Canales",
			'45057' => "Chueca",
			'45050' => "Ciruelos",
			'45051' => "Cobeja",
			'45052' => "Cobisa",
			'45053' => "Consuegra",
			'45054' => "Corral de Almaguer",
			'45055' => "Cuerva",
			'45058' => "Domingo Pérez",
			'45059' => "Dosbarrios",
			'45060' => "Erustes",
			'45061' => "Escalona",
			'45062' => "Escalonilla",
			'45063' => "Espinoso del Rey",
			'45064' => "Esquivias",
			'45065' => "Estrella, La",
			'45066' => "Fuensalida",
			'45067' => "Gálvez",
			'45068' => "Garciotum",
			'45069' => "Gerindote",
			'45070' => "Guadamur",
			'45071' => "Guardia, La",
			'45072' => "Herencias, Las",
			'45073' => "Herreruela de Oropesa",
			'45074' => "Hinojosa de San Vicente",
			'45075' => "Hontanar",
			'45076' => "Hormigos",
			'45077' => "Huecas",
			'45078' => "Huerta de Valdecarábanos",
			'45079' => "Iglesuela del Tiétar, La",
			'45080' => "Illán de Vacas",
			'45081' => "Illescas",
			'45082' => "Lagartera",
			'45083' => "Layos",
			'45084' => "Lillo",
			'45085' => "Lominchar",
			'45086' => "Lucillos",
			'45087' => "Madridejos",
			'45088' => "Magán",
			'45089' => "Malpica de Tajo",
			'45090' => "Manzaneque",
			'45091' => "Maqueda",
			'45092' => "Marjaliza",
			'45093' => "Marrupe",
			'45094' => "Mascaraque",
			'45095' => "Mata, La",
			'45096' => "Mazarambroz",
			'45097' => "Mejorada",
			'45098' => "Menasalbas",
			'45099' => "Méntrida",
			'45100' => "Mesegar de Tajo",
			'45101' => "Miguel Esteban",
			'45102' => "Mocejón",
			'45103' => "Mohedas de la Jara",
			'45104' => "Montearagón",
			'45105' => "Montesclaros",
			'45106' => "Mora",
			'45107' => "Nambroca",
			'45108' => "Nava de Ricomalillo, La",
			'45109' => "Navahermosa",
			'45110' => "Navalcán",
			'45111' => "Navalmoralejo",
			'45112' => "Navalmorales, Los",
			'45113' => "Navalucillos, Los",
			'45114' => "Navamorcuende",
			'45115' => "Noblejas",
			'45116' => "Noez",
			'45117' => "Nombela",
			'45118' => "Novés",
			'45119' => "Numancia de la Sagra",
			'45120' => "Nuño Gómez",
			'45121' => "Ocaña",
			'45122' => "Olías del Rey",
			'45123' => "Ontígola",
			'45124' => "Orgaz",
			'45125' => "Oropesa",
			'45126' => "Otero",
			'45127' => "Palomeque",
			'45128' => "Pantoja",
			'45129' => "Paredes de Escalona",
			'45130' => "Parrillas",
			'45131' => "Pelahustán",
			'45132' => "Pepino",
			'45133' => "Polán",
			'45134' => "Portillo de Toledo",
			'45135' => "Puebla de Almoradiel, La",
			'45136' => "Puebla de Montalbán, La",
			'45137' => "Pueblanueva, La",
			'45138' => "Puente del Arzobispo, El",
			'45139' => "Puerto de San Vicente",
			'45140' => "Pulgar",
			'45141' => "Quero",
			'45142' => "Quintanar de la Orden",
			'45143' => "Quismondo",
			'45144' => "Real de San Vicente, El",
			'45145' => "Recas",
			'45146' => "Retamoso de la Jara",
			'45147' => "Rielves",
			'45148' => "Robledo del Mazo",
			'45149' => "Romeral, El",
			'45150' => "San Bartolomé de las Abiertas",
			'45151' => "San Martín de Montalbán",
			'45152' => "San Martín de Pusa",
			'45153' => "San Pablo de los Montes",
			'45154' => "San Román de los Montes",
			'45155' => "Santa Ana de Pusa",
			'45156' => "Santa Cruz de la Zarza",
			'45157' => "Santa Cruz del Retamar",
			'45158' => "Santa Olalla",
			'45901' => "Santo Domingo-Caudilla",
			'45159' => "Sartajada",
			'45160' => "Segurilla",
			'45161' => "Seseña",
			'45162' => "Sevilleja de la Jara",
			'45163' => "Sonseca",
			'45164' => "Sotillo de las Palomas",
			'45165' => "Talavera de la Reina",
			'45166' => "Tembleque",
			'45167' => "Toboso, El",
			'45168' => "Toledo",
			'45169' => "Torralba de Oropesa",
			'45171' => "Torre de Esteban Hambrán, La",
			'45170' => "Torrecilla de la Jara",
			'45172' => "Torrico",
			'45173' => "Torrijos",
			'45174' => "Totanés",
			'45175' => "Turleque",
			'45176' => "Ugena",
			'45177' => "Urda",
			'45179' => "Valdeverdeja",
			'45180' => "Valmojado",
			'45181' => "Velada",
			'45182' => "Ventas con Peña Aguilera, Las",
			'45183' => "Ventas de Retamosa, Las",
			'45184' => "Ventas de San Julián, Las",
			'45186' => "Villa de Don Fadrique, La",
			'45185' => "Villacañas",
			'45187' => "Villafranca de los Caballeros",
			'45188' => "Villaluenga de la Sagra",
			'45189' => "Villamiel de Toledo",
			'45190' => "Villaminaya",
			'45191' => "Villamuelas",
			'45192' => "Villanueva de Alcardete",
			'45193' => "Villanueva de Bogas",
			'45194' => "Villarejo de Montalbán",
			'45195' => "Villarrubia de Santiago",
			'45196' => "Villaseca de la Sagra",
			'45197' => "Villasequilla",
			'45198' => "Villatobas",
			'45199' => "Viso de San Juan, El",
			'45200' => "Yébenes, Los",
			'45201' => "Yeles",
			'45202' => "Yepes",
			'45203' => "Yuncler",
			'45204' => "Yunclillos",
			'45205' => "Yuncos",
			'46001' => "Ademuz",
			'46002' => "Ador",
			'46004' => "Agullent",
			'46042' => "Aielo de Malferit",
			'46043' => "Aielo de Rugat",
			'46005' => "Alaquàs",
			'46006' => "Albaida",
			'46007' => "Albal",
			'46008' => "Albalat de la Ribera",
			'46009' => "Albalat dels Sorells",
			'46010' => "Albalat dels Tarongers",
			'46011' => "Alberic",
			'46012' => "Alborache",
			'46013' => "Alboraia/Alboraya",
			'46014' => "Albuixech",
			'46016' => "Alcàntera de Xúquer",
			'46015' => "Alcàsser",
			'46018' => "Alcublas",
			'46020' => "Alcúdia de Crespins, l'",
			'46019' => "Alcúdia, l'",
			'46021' => "Aldaia",
			'46022' => "Alfafar",
			'46024' => "Alfara de la Baronia",
			'46025' => "Alfara del Patriarca",
			'46026' => "Alfarb",
			'46027' => "Alfarrasí",
			'46023' => "Alfauir",
			'46028' => "Algar de Palància",
			'46029' => "Algemesí",
			'46030' => "Algímia d'Alfara",
			'46031' => "Alginet",
			'46032' => "Almàssera",
			'46033' => "Almiserà",
			'46034' => "Almoines",
			'46035' => "Almussafes",
			'46036' => "Alpuente",
			'46037' => "Alqueria de la Comtessa, l'",
			'46017' => "Alzira",
			'46038' => "Andilla",
			'46039' => "Anna",
			'46040' => "Antella",
			'46041' => "Aras de los Olmos",
			'46003' => "Atzeneta d'Albaida",
			'46044' => "Ayora",
			'46046' => "Barx",
			'46045' => "Barxeta",
			'46047' => "Bèlgida",
			'46048' => "Bellreguard",
			'46049' => "Bellús",
			'46050' => "Benagéber",
			'46051' => "Benaguasil",
			'46052' => "Benavites",
			'46053' => "Beneixida",
			'46054' => "Benetússer",
			'46055' => "Beniarjó",
			'46056' => "Beniatjar",
			'46057' => "Benicolet",
			'46904' => "Benicull de Xúquer",
			'46060' => "Benifaió",
			'46059' => "Benifairó de la Valldigna",
			'46058' => "Benifairó de les Valls",
			'46061' => "Beniflá",
			'46062' => "Benigànim",
			'46063' => "Benimodo",
			'46064' => "Benimuslem",
			'46065' => "Beniparrell",
			'46066' => "Benirredrà",
			'46067' => "Benissanó",
			'46068' => "Benissoda",
			'46069' => "Benissuera",
			'46070' => "Bétera",
			'46071' => "Bicorp",
			'46072' => "Bocairent",
			'46073' => "Bolbaite",
			'46074' => "Bonrepòs i Mirambell",
			'46075' => "Bufali",
			'46076' => "Bugarra",
			'46077' => "Buñol",
			'46078' => "Burjassot",
			'46079' => "Calles",
			'46080' => "Camporrobles",
			'46081' => "Canals",
			'46082' => "Canet d'En Berenguer",
			'46083' => "Carcaixent",
			'46084' => "Càrcer",
			'46085' => "Carlet",
			'46086' => "Carrícola",
			'46087' => "Casas Altas",
			'46088' => "Casas Bajas",
			'46089' => "Casinos",
			'46257' => "Castelló",
			'46090' => "Castelló de Rugat",
			'46091' => "Castellonet de la Conquesta",
			'46092' => "Castielfabib",
			'46093' => "Catadau",
			'46094' => "Catarroja",
			'46095' => "Caudete de las Fuentes",
			'46096' => "Cerdà",
			'46107' => "Chella",
			'46106' => "Chelva",
			'46108' => "Chera",
			'46109' => "Cheste",
			'46111' => "Chiva",
			'46112' => "Chulilla",
			'46097' => "Cofrentes",
			'46098' => "Corbera",
			'46099' => "Cortes de Pallás",
			'46100' => "Cotes",
			'46105' => "Cullera",
			'46113' => "Daimús",
			'46114' => "Domeño",
			'46115' => "Dos Aguas",
			'46116' => "Eliana, l'",
			'46117' => "Emperador",
			'46118' => "Enguera",
			'46119' => "Énova, l'",
			'46120' => "Estivella",
			'46121' => "Estubeny",
			'46122' => "Faura",
			'46123' => "Favara",
			'46126' => "Foios",
			'46128' => "Font de la Figuera, la",
			'46127' => "Font d'en Carròs, la",
			'46124' => "Fontanars dels Alforins",
			'46125' => "Fortaleny",
			'46129' => "Fuenterrobles",
			'46131' => "Gandia",
			'46902' => "Gátova",
			'46130' => "Gavarda",
			'46132' => "Genovés, el",
			'46133' => "Gestalgar",
			'46134' => "Gilet",
			'46135' => "Godella",
			'46136' => "Godelleta",
			'46137' => "Granja de la Costera, la",
			'46138' => "Guadasséquies",
			'46139' => "Guadassuar",
			'46140' => "Guardamar de la Safor",
			'46141' => "Higueruelas",
			'46142' => "Jalance",
			'46144' => "Jarafuel",
			'46154' => "Llanera de Ranes",
			'46155' => "Llaurí",
			'46147' => "Llíria",
			'46152' => "Llocnou de la Corona",
			'46153' => "Llocnou de Sant Jeroni",
			'46151' => "Llocnou d'En Fenollet",
			'46156' => "Llombai",
			'46157' => "Llosa de Ranes, la",
			'46150' => "Llutxent",
			'46148' => "Loriguilla",
			'46149' => "Losa del Obispo",
			'46158' => "Macastre",
			'46159' => "Manises",
			'46160' => "Manuel",
			'46161' => "Marines",
			'46162' => "Massalavés",
			'46163' => "Massalfassar",
			'46164' => "Massamagrell",
			'46165' => "Massanassa",
			'46166' => "Meliana",
			'46167' => "Millares",
			'46168' => "Miramar",
			'46169' => "Mislata",
			'46170' => "Moixent/Mogente",
			'46171' => "Moncada",
			'46173' => "Montaverner",
			'46174' => "Montesa",
			'46175' => "Montitxelvo/Montichelvo",
			'46176' => "Montroi/Montroy",
			'46172' => "Montserrat",
			'46177' => "Museros",
			'46178' => "Nàquera/Náquera",
			'46179' => "Navarrés",
			'46180' => "Novetlè",
			'46181' => "Oliva",
			'46183' => "Olleria, l'",
			'46182' => "Olocau",
			'46184' => "Ontinyent",
			'46185' => "Otos",
			'46186' => "Paiporta",
			'46187' => "Palma de Gandía",
			'46188' => "Palmera",
			'46189' => "Palomar, el",
			'46190' => "Paterna",
			'46191' => "Pedralba",
			'46192' => "Petrés",
			'46193' => "Picanya",
			'46194' => "Picassent",
			'46195' => "Piles",
			'46196' => "Pinet",
			'46199' => "Pobla de Farnals, la",
			'46202' => "Pobla de Vallbona, la",
			'46200' => "Pobla del Duc, la",
			'46203' => "Pobla Llarga, la",
			'46197' => "Polinyà de Xúquer",
			'46198' => "Potries",
			'46205' => "Puçol",
			'46201' => "Puebla de San Miguel",
			'46204' => "Puig de Santa Maria, el",
			'46101' => "Quart de les Valls",
			'46102' => "Quart de Poblet",
			'46103' => "Quartell",
			'46104' => "Quatretonda",
			'46206' => "Quesa",
			'46207' => "Rafelbunyol",
			'46208' => "Rafelcofer",
			'46209' => "Rafelguaraf",
			'46210' => "Ráfol de Salem",
			'46212' => "Real",
			'46211' => "Real de Gandia, el",
			'46213' => "Requena",
			'46214' => "Riba-roja de Túria",
			'46215' => "Riola",
			'46216' => "Rocafort",
			'46217' => "Rotglà i Corberà",
			'46218' => "Ròtova",
			'46219' => "Rugat",
			'46220' => "Sagunt/Sagunto",
			'46221' => "Salem",
			'46903' => "San Antonio de Benagéber",
			'46222' => "Sant Joanet",
			'46223' => "Sedaví",
			'46224' => "Segart",
			'46225' => "Sellent",
			'46226' => "Sempere",
			'46227' => "Senyera",
			'46228' => "Serra",
			'46229' => "Siete Aguas",
			'46230' => "Silla",
			'46231' => "Simat de la Valldigna",
			'46232' => "Sinarcas",
			'46233' => "Sollana",
			'46234' => "Sot de Chera",
			'46235' => "Sueca",
			'46236' => "Sumacàrcer",
			'46237' => "Tavernes Blanques",
			'46238' => "Tavernes de la Valldigna",
			'46239' => "Teresa de Cofrentes",
			'46240' => "Terrateig",
			'46241' => "Titaguas",
			'46242' => "Torrebaja",
			'46243' => "Torrella",
			'46244' => "Torrent",
			'46245' => "Torres Torres",
			'46246' => "Tous",
			'46247' => "Tuéjar",
			'46248' => "Turís",
			'46249' => "Utiel",
			'46250' => "València",
			'46251' => "Vallada",
			'46252' => "Vallanca",
			'46253' => "Vallés",
			'46254' => "Venta del Moro",
			'46255' => "Vilallonga/Villalonga",
			'46256' => "Vilamarxant",
			'46258' => "Villar del Arzobispo",
			'46259' => "Villargordo del Cabriel",
			'46260' => "Vinalesa",
			'46145' => "Xàtiva",
			'46143' => "Xeraco",
			'46146' => "Xeresa",
			'46110' => "Xirivella",
			'46261' => "Yátova",
			'46262' => "Yesa, La",
			'46263' => "Zarra",
			'47001' => "Adalia",
			'47002' => "Aguasal",
			'47003' => "Aguilar de Campos",
			'47004' => "Alaejos",
			'47005' => "Alcazarén",
			'47006' => "Aldea de San Miguel",
			'47007' => "Aldeamayor de San Martín",
			'47008' => "Almenara de Adaja",
			'47009' => "Amusquillo",
			'47010' => "Arroyo de la Encomienda",
			'47011' => "Ataquines",
			'47012' => "Bahabón",
			'47013' => "Barcial de la Loma",
			'47014' => "Barruelo del Valle",
			'47015' => "Becilla de Valderaduey",
			'47016' => "Benafarces",
			'47017' => "Bercero",
			'47018' => "Berceruelo",
			'47019' => "Berrueces",
			'47020' => "Bobadilla del Campo",
			'47021' => "Bocigas",
			'47022' => "Bocos de Duero",
			'47023' => "Boecillo",
			'47024' => "Bolaños de Campos",
			'47025' => "Brahojos de Medina",
			'47026' => "Bustillo de Chaves",
			'47027' => "Cabezón de Pisuerga",
			'47028' => "Cabezón de Valderaduey",
			'47029' => "Cabreros del Monte",
			'47030' => "Campaspero",
			'47031' => "Campillo, El",
			'47032' => "Camporredondo",
			'47033' => "Canalejas de Peñafiel",
			'47034' => "Canillas de Esgueva",
			'47035' => "Carpio",
			'47036' => "Casasola de Arión",
			'47037' => "Castrejón de Trabancos",
			'47038' => "Castrillo de Duero",
			'47039' => "Castrillo-Tejeriego",
			'47040' => "Castrobol",
			'47041' => "Castrodeza",
			'47042' => "Castromembibre",
			'47043' => "Castromonte",
			'47044' => "Castronuevo de Esgueva",
			'47045' => "Castronuño",
			'47046' => "Castroponce",
			'47047' => "Castroverde de Cerrato",
			'47048' => "Ceinos de Campos",
			'47049' => "Cervillego de la Cruz",
			'47050' => "Cigales",
			'47051' => "Ciguñuela",
			'47052' => "Cistérniga",
			'47053' => "Cogeces de Íscar",
			'47054' => "Cogeces del Monte",
			'47055' => "Corcos",
			'47056' => "Corrales de Duero",
			'47057' => "Cubillas de Santa Marta",
			'47058' => "Cuenca de Campos",
			'47059' => "Curiel de Duero",
			'47060' => "Encinas de Esgueva",
			'47061' => "Esguevillas de Esgueva",
			'47062' => "Fombellida",
			'47063' => "Fompedraza",
			'47064' => "Fontihoyuelo",
			'47065' => "Fresno el Viejo",
			'47066' => "Fuensaldaña",
			'47067' => "Fuente el Sol",
			'47068' => "Fuente-Olmedo",
			'47069' => "Gallegos de Hornija",
			'47070' => "Gatón de Campos",
			'47071' => "Geria",
			'47073' => "Herrín de Campos",
			'47074' => "Hornillos de Eresma",
			'47075' => "Íscar",
			'47076' => "Laguna de Duero",
			'47077' => "Langayo",
			'47079' => "Llano de Olmedo",
			'47078' => "Lomoviejo",
			'47080' => "Manzanillo",
			'47081' => "Marzales",
			'47082' => "Matapozuelos",
			'47083' => "Matilla de los Caños",
			'47084' => "Mayorga",
			'47086' => "Medina de Rioseco",
			'47085' => "Medina del Campo",
			'47087' => "Megeces",
			'47088' => "Melgar de Abajo",
			'47089' => "Melgar de Arriba",
			'47090' => "Mojados",
			'47091' => "Monasterio de Vega",
			'47092' => "Montealegre de Campos",
			'47093' => "Montemayor de Pililla",
			'47094' => "Moral de la Reina",
			'47095' => "Moraleja de las Panaderas",
			'47096' => "Morales de Campos",
			'47097' => "Mota del Marqués",
			'47098' => "Mucientes",
			'47099' => "Mudarra, La",
			'47100' => "Muriel",
			'47101' => "Nava del Rey",
			'47102' => "Nueva Villa de las Torres",
			'47103' => "Olivares de Duero",
			'47104' => "Olmedo",
			'47105' => "Olmos de Esgueva",
			'47106' => "Olmos de Peñafiel",
			'47109' => "Palazuelo de Vedija",
			'47110' => "Parrilla, La",
			'47111' => "Pedraja de Portillo, La",
			'47112' => "Pedrajas de San Esteban",
			'47113' => "Pedrosa del Rey",
			'47114' => "Peñafiel",
			'47115' => "Peñaflor de Hornija",
			'47116' => "Pesquera de Duero",
			'47117' => "Piña de Esgueva",
			'47118' => "Piñel de Abajo",
			'47119' => "Piñel de Arriba",
			'47121' => "Pollos",
			'47122' => "Portillo",
			'47123' => "Pozal de Gallinas",
			'47124' => "Pozaldez",
			'47125' => "Pozuelo de la Orden",
			'47126' => "Puras",
			'47127' => "Quintanilla de Arriba",
			'47129' => "Quintanilla de Onésimo",
			'47130' => "Quintanilla de Trigueros",
			'47128' => "Quintanilla del Molar",
			'47131' => "Rábano",
			'47132' => "Ramiro",
			'47133' => "Renedo de Esgueva",
			'47134' => "Roales de Campos",
			'47135' => "Robladillo",
			'47137' => "Roturas",
			'47138' => "Rubí de Bracamonte",
			'47139' => "Rueda",
			'47140' => "Saelices de Mayorga",
			'47141' => "Salvador de Zapardiel",
			'47142' => "San Cebrián de Mazote",
			'47143' => "San Llorente",
			'47144' => "San Martín de Valvení",
			'47145' => "San Miguel del Arroyo",
			'47146' => "San Miguel del Pino",
			'47147' => "San Pablo de la Moraleja",
			'47148' => "San Pedro de Latarce",
			'47149' => "San Pelayo",
			'47150' => "San Román de Hornija",
			'47151' => "San Salvador",
			'47156' => "San Vicente del Palacio",
			'47152' => "Santa Eufemia del Arroyo",
			'47153' => "Santervás de Campos",
			'47154' => "Santibáñez de Valcorba",
			'47155' => "Santovenia de Pisuerga",
			'47157' => "Sardón de Duero",
			'47158' => "Seca, La",
			'47159' => "Serrada",
			'47160' => "Siete Iglesias de Trabancos",
			'47161' => "Simancas",
			'47162' => "Tamariz de Campos",
			'47163' => "Tiedra",
			'47164' => "Tordehumos",
			'47165' => "Tordesillas",
			'47169' => "Torre de Esgueva",
			'47170' => "Torre de Peñafiel",
			'47166' => "Torrecilla de la Abadesa",
			'47167' => "Torrecilla de la Orden",
			'47168' => "Torrecilla de la Torre",
			'47171' => "Torrelobatón",
			'47172' => "Torrescárcela",
			'47173' => "Traspinedo",
			'47174' => "Trigueros del Valle",
			'47175' => "Tudela de Duero",
			'47176' => "Unión de Campos, La",
			'47177' => "Urones de Castroponce",
			'47178' => "Urueña",
			'47179' => "Valbuena de Duero",
			'47180' => "Valdearcos de la Vega",
			'47181' => "Valdenebro de los Valles",
			'47182' => "Valdestillas",
			'47183' => "Valdunquillo",
			'47186' => "Valladolid",
			'47184' => "Valoria la Buena",
			'47185' => "Valverde de Campos",
			'47187' => "Vega de Ruiponce",
			'47188' => "Vega de Valdetronco",
			'47189' => "Velascálvaro",
			'47190' => "Velilla",
			'47191' => "Velliza",
			'47192' => "Ventosa de la Cuesta",
			'47193' => "Viana de Cega",
			'47195' => "Villabáñez",
			'47196' => "Villabaruz de Campos",
			'47197' => "Villabrágima",
			'47198' => "Villacarralón",
			'47199' => "Villacid de Campos",
			'47200' => "Villaco",
			'47203' => "Villafrades de Campos",
			'47204' => "Villafranca de Duero",
			'47205' => "Villafrechós",
			'47206' => "Villafuerte",
			'47207' => "Villagarcía de Campos",
			'47208' => "Villagómez la Nueva",
			'47209' => "Villalán de Campos",
			'47210' => "Villalar de los Comuneros",
			'47211' => "Villalba de la Loma",
			'47212' => "Villalba de los Alcores",
			'47213' => "Villalbarba",
			'47214' => "Villalón de Campos",
			'47215' => "Villamuriel de Campos",
			'47216' => "Villán de Tordesillas",
			'47217' => "Villanubla",
			'47218' => "Villanueva de Duero",
			'47219' => "Villanueva de la Condesa",
			'47220' => "Villanueva de los Caballeros",
			'47221' => "Villanueva de los Infantes",
			'47222' => "Villanueva de San Mancio",
			'47223' => "Villardefrades",
			'47224' => "Villarmentero de Esgueva",
			'47225' => "Villasexmir",
			'47226' => "Villavaquerín",
			'47227' => "Villavellid",
			'47228' => "Villaverde de Medina",
			'47229' => "Villavicencio de los Caballeros",
			'47194' => "Viloria",
			'47230' => "Wamba",
			'47231' => "Zaratán",
			'47232' => "Zarza, La",
			'48001' => "Abadiño",
			'48002' => "Abanto y Ciérvana-Abanto Zierbena",
			'48911' => "Ajangiz",
			'48912' => "Alonsotegi",
			'48003' => "Amorebieta-Etxano",
			'48004' => "Amoroto",
			'48005' => "Arakaldo",
			'48006' => "Arantzazu",
			'48093' => "Areatza",
			'48009' => "Arrankudiaga-Zollo",
			'48914' => "Arratzu",
			'48010' => "Arrieta",
			'48011' => "Arrigorriaga",
			'48023' => "Artea",
			'48008' => "Artzentales",
			'48091' => "Atxondo",
			'48070' => "Aulesti",
			'48012' => "Bakio",
			'48090' => "Balmaseda",
			'48013' => "Barakaldo",
			'48014' => "Barrika",
			'48015' => "Basauri",
			'48092' => "Bedia",
			'48016' => "Berango",
			'48017' => "Bermeo",
			'48018' => "Berriatua",
			'48019' => "Berriz",
			'48020' => "Bilbao",
			'48021' => "Busturia",
			'48901' => "Derio",
			'48026' => "Dima",
			'48027' => "Durango",
			'48028' => "Ea",
			'48031' => "Elantxobe",
			'48032' => "Elorrio",
			'48902' => "Erandio",
			'48033' => "Ereño",
			'48034' => "Ermua",
			'48079' => "Errigoiti",
			'48029' => "Etxebarri",
			'48030' => "Etxebarria",
			'48906' => "Forua",
			'48035' => "Fruiz",
			'48036' => "Galdakao",
			'48037' => "Galdames",
			'48038' => "Gamiz-Fika",
			'48039' => "Garai",
			'48040' => "Gatika",
			'48041' => "Gautegiz Arteaga",
			'48046' => "Gernika-Lumo",
			'48044' => "Getxo",
			'48047' => "Gizaburuaga",
			'48042' => "Gordexola",
			'48043' => "Gorliz",
			'48045' => "Güeñes",
			'48048' => "Ibarrangelu",
			'48094' => "Igorre",
			'48049' => "Ispaster",
			'48910' => "Iurreta",
			'48050' => "Izurtza",
			'48022' => "Karrantza Harana/Valle de Carranza",
			'48907' => "Kortezubi",
			'48051' => "Lanestosa",
			'48052' => "Larrabetzu",
			'48053' => "Laukiz",
			'48054' => "Leioa",
			'48057' => "Lekeitio",
			'48055' => "Lemoa",
			'48056' => "Lemoiz",
			'48081' => "Lezama",
			'48903' => "Loiu",
			'48058' => "Mallabia",
			'48059' => "Mañaria",
			'48060' => "Markina-Xemein",
			'48061' => "Maruri-Jatabe",
			'48062' => "Mendata",
			'48063' => "Mendexa",
			'48064' => "Meñaka",
			'48066' => "Morga",
			'48068' => "Mundaka",
			'48069' => "Mungia",
			'48007' => "Munitibar-Arbatzegi Gerrikaitz",
			'48908' => "Murueta",
			'48071' => "Muskiz",
			'48067' => "Muxika",
			'48909' => "Nabarniz",
			'48073' => "Ondarroa",
			'48075' => "Orozko",
			'48083' => "Ortuella",
			'48072' => "Otxandio",
			'48077' => "Plentzia",
			'48078' => "Portugalete",
			'48082' => "Santurtzi",
			'48084' => "Sestao",
			'48904' => "Sondika",
			'48085' => "Sopela",
			'48086' => "Sopuerta",
			'48076' => "Sukarrieta",
			'48087' => "Trucios-Turtzioz",
			'48088' => "Ubide",
			'48065' => "Ugao-Miraballes",
			'48089' => "Urduliz",
			'48074' => "Urduña/Orduña",
			'48916' => "Usansolo",
			'48080' => "Valle de Trápaga-Trapagaran",
			'48095' => "Zaldibar",
			'48096' => "Zalla",
			'48905' => "Zamudio",
			'48097' => "Zaratamo",
			'48024' => "Zeanuri",
			'48025' => "Zeberio",
			'48913' => "Zierbena",
			'48915' => "Ziortza-Bolibar",
			'49002' => "Abezames",
			'49003' => "Alcañices",
			'49004' => "Alcubilla de Nogales",
			'49005' => "Alfaraz de Sayago",
			'49006' => "Algodre",
			'49007' => "Almaraz de Duero",
			'49008' => "Almeida de Sayago",
			'49009' => "Andavías",
			'49010' => "Arcenillas",
			'49011' => "Arcos de la Polvorosa",
			'49012' => "Argañín",
			'49013' => "Argujillo",
			'49014' => "Arquillinos",
			'49015' => "Arrabalde",
			'49016' => "Aspariegos",
			'49017' => "Asturianos",
			'49018' => "Ayoó de Vidriales",
			'49019' => "Barcial del Barco",
			'49020' => "Belver de los Montes",
			'49021' => "Benavente",
			'49022' => "Benegiles",
			'49023' => "Bermillo de Sayago",
			'49024' => "Bóveda de Toro, La",
			'49025' => "Bretó",
			'49026' => "Bretocino",
			'49027' => "Brime de Sog",
			'49028' => "Brime de Urz",
			'49029' => "Burganes de Valverde",
			'49030' => "Bustillo del Oro",
			'49031' => "Cabañas de Sayago",
			'49032' => "Calzadilla de Tera",
			'49033' => "Camarzana de Tera",
			'49034' => "Cañizal",
			'49035' => "Cañizo",
			'49036' => "Carbajales de Alba",
			'49037' => "Carbellino",
			'49038' => "Casaseca de Campeán",
			'49039' => "Casaseca de las Chanas",
			'49040' => "Castrillo de la Guareña",
			'49041' => "Castrogonzalo",
			'49042' => "Castronuevo",
			'49043' => "Castroverde de Campos",
			'49044' => "Cazurra",
			'49046' => "Cerecinos de Campos",
			'49047' => "Cerecinos del Carrizal",
			'49048' => "Cernadilla",
			'49050' => "Cobreros",
			'49052' => "Coomonte",
			'49053' => "Coreses",
			'49054' => "Corrales del Vino",
			'49055' => "Cotanes del Monte",
			'49056' => "Cubillos",
			'49057' => "Cubo de Benavente",
			'49058' => "Cubo de Tierra del Vino, El",
			'49059' => "Cuelgamures",
			'49061' => "Entrala",
			'49062' => "Espadañedo",
			'49063' => "Faramontanos de Tábara",
			'49064' => "Fariza",
			'49065' => "Fermoselle",
			'49066' => "Ferreras de Abajo",
			'49067' => "Ferreras de Arriba",
			'49068' => "Ferreruela",
			'49069' => "Figueruela de Arriba",
			'49071' => "Fonfría",
			'49075' => "Fresno de la Polvorosa",
			'49076' => "Fresno de la Ribera",
			'49077' => "Fresno de Sayago",
			'49078' => "Friera de Valverde",
			'49079' => "Fuente Encalada",
			'49080' => "Fuentelapeña",
			'49082' => "Fuentes de Ropel",
			'49081' => "Fuentesaúco",
			'49083' => "Fuentesecas",
			'49084' => "Fuentespreadas",
			'49085' => "Galende",
			'49086' => "Gallegos del Pan",
			'49087' => "Gallegos del Río",
			'49088' => "Gamones",
			'49090' => "Gema",
			'49091' => "Granja de Moreruela",
			'49092' => "Granucillo",
			'49093' => "Guarrate",
			'49094' => "Hermisende",
			'49095' => "Hiniesta, La",
			'49096' => "Jambrina",
			'49097' => "Justel",
			'49098' => "Losacino",
			'49099' => "Losacio",
			'49100' => "Lubián",
			'49101' => "Luelmo",
			'49102' => "Maderal, El",
			'49103' => "Madridanos",
			'49104' => "Mahide",
			'49105' => "Maire de Castroponce",
			'49107' => "Malva",
			'49108' => "Manganeses de la Lampreana",
			'49109' => "Manganeses de la Polvorosa",
			'49110' => "Manzanal de Arriba",
			'49112' => "Manzanal de los Infantes",
			'49111' => "Manzanal del Barco",
			'49113' => "Matilla de Arzón",
			'49114' => "Matilla la Seca",
			'49115' => "Mayalde",
			'49116' => "Melgar de Tera",
			'49117' => "Micereces de Tera",
			'49118' => "Milles de la Polvorosa",
			'49119' => "Molacillos",
			'49120' => "Molezuelas de la Carballeda",
			'49121' => "Mombuey",
			'49122' => "Monfarracinos",
			'49123' => "Montamarta",
			'49124' => "Moral de Sayago",
			'49126' => "Moraleja de Sayago",
			'49125' => "Moraleja del Vino",
			'49128' => "Morales de Rey",
			'49129' => "Morales de Toro",
			'49130' => "Morales de Valverde",
			'49127' => "Morales del Vino",
			'49131' => "Moralina",
			'49132' => "Moreruela de los Infanzones",
			'49133' => "Moreruela de Tábara",
			'49134' => "Muelas de los Caballeros",
			'49135' => "Muelas del Pan",
			'49136' => "Muga de Sayago",
			'49137' => "Navianos de Valverde",
			'49138' => "Olmillos de Castro",
			'49139' => "Otero de Bodas",
			'49141' => "Pajares de la Lampreana",
			'49143' => "Palacios de Sanabria",
			'49142' => "Palacios del Pan",
			'49145' => "Pedralba de la Pradería",
			'49146' => "Pego, El",
			'49147' => "Peleagonzalo",
			'49148' => "Peleas de Abajo",
			'49149' => "Peñausende",
			'49150' => "Peque",
			'49151' => "Perdigón, El",
			'49152' => "Pereruela",
			'49153' => "Perilla de Castro",
			'49154' => "Pías",
			'49155' => "Piedrahita de Castro",
			'49156' => "Pinilla de Toro",
			'49157' => "Pino del Oro",
			'49158' => "Piñero, El",
			'49160' => "Pobladura de Valderaduey",
			'49159' => "Pobladura del Valle",
			'49162' => "Porto",
			'49163' => "Pozoantiguo",
			'49164' => "Pozuelo de Tábara",
			'49165' => "Prado",
			'49166' => "Puebla de Sanabria",
			'49167' => "Pueblica de Valverde",
			'49170' => "Quintanilla de Urz",
			'49168' => "Quintanilla del Monte",
			'49169' => "Quintanilla del Olmo",
			'49171' => "Quiruelas de Vidriales",
			'49172' => "Rabanales",
			'49173' => "Rábano de Aliste",
			'49174' => "Requejo",
			'49175' => "Revellinos",
			'49176' => "Riofrío de Aliste",
			'49177' => "Rionegro del Puente",
			'49178' => "Roales",
			'49179' => "Robleda-Cervantes",
			'49180' => "Roelos de Sayago",
			'49181' => "Rosinos de la Requejada",
			'49183' => "Salce",
			'49184' => "Samir de los Caños",
			'49185' => "San Agustín del Pozo",
			'49186' => "San Cebrián de Castro",
			'49187' => "San Cristóbal de Entreviñas",
			'49188' => "San Esteban del Molar",
			'49189' => "San Justo",
			'49190' => "San Martín de Valderaduey",
			'49191' => "San Miguel de la Ribera",
			'49192' => "San Miguel del Valle",
			'49193' => "San Pedro de Ceque",
			'49194' => "San Pedro de la Nave-Almendra",
			'49208' => "San Vicente de la Cabeza",
			'49209' => "San Vitero",
			'49197' => "Santa Clara de Avedillo",
			'49199' => "Santa Colomba de las Monjas",
			'49200' => "Santa Cristina de la Polvorosa",
			'49201' => "Santa Croya de Tera",
			'49202' => "Santa Eufemia del Barco",
			'49203' => "Santa María de la Vega",
			'49204' => "Santa María de Valverde",
			'49205' => "Santibáñez de Tera",
			'49206' => "Santibáñez de Vidriales",
			'49207' => "Santovenia",
			'49210' => "Sanzoles",
			'49214' => "Tábara",
			'49216' => "Tapioles",
			'49219' => "Toro",
			'49220' => "Torre del Valle, La",
			'49221' => "Torregamones",
			'49222' => "Torres del Carrizal",
			'49223' => "Trabazos",
			'49224' => "Trefacio",
			'49225' => "Uña de Quintana",
			'49226' => "Vadillo de la Guareña",
			'49227' => "Valcabado",
			'49228' => "Valdefinjas",
			'49229' => "Valdescorriel",
			'49230' => "Vallesa de la Guareña",
			'49231' => "Vega de Tera",
			'49232' => "Vega de Villalobos",
			'49233' => "Vegalatrave",
			'49234' => "Venialbo",
			'49235' => "Vezdemarbán",
			'49236' => "Vidayanes",
			'49237' => "Videmala",
			'49238' => "Villabrázaro",
			'49239' => "Villabuena del Puente",
			'49240' => "Villadepera",
			'49241' => "Villaescusa",
			'49242' => "Villafáfila",
			'49243' => "Villaferrueña",
			'49244' => "Villageriz",
			'49245' => "Villalazán",
			'49246' => "Villalba de la Lampreana",
			'49247' => "Villalcampo",
			'49248' => "Villalobos",
			'49249' => "Villalonso",
			'49250' => "Villalpando",
			'49251' => "Villalube",
			'49252' => "Villamayor de Campos",
			'49255' => "Villamor de los Escuderos",
			'49256' => "Villanázar",
			'49257' => "Villanueva de Azoague",
			'49258' => "Villanueva de Campeán",
			'49259' => "Villanueva de las Peras",
			'49260' => "Villanueva del Campo",
			'49263' => "Villar de Fallaves",
			'49264' => "Villar del Buey",
			'49261' => "Villaralbo",
			'49262' => "Villardeciervos",
			'49265' => "Villardiegua de la Ribera",
			'49266' => "Villárdiga",
			'49267' => "Villardondiego",
			'49268' => "Villarrín de Campos",
			'49269' => "Villaseco del Pan",
			'49270' => "Villavendimio",
			'49272' => "Villaveza de Valverde",
			'49271' => "Villaveza del Agua",
			'49273' => "Viñas",
			'49275' => "Zamora",
			'50001' => "Abanto",
			'50002' => "Acered",
			'50003' => "Agón",
			'50004' => "Aguarón",
			'50005' => "Aguilón",
			'50006' => "Ainzón",
			'50007' => "Aladrén",
			'50008' => "Alagón",
			'50009' => "Alarba",
			'50010' => "Alberite de San Juan",
			'50011' => "Albeta",
			'50012' => "Alborge",
			'50013' => "Alcalá de Ebro",
			'50014' => "Alcalá de Moncayo",
			'50015' => "Alconchel de Ariza",
			'50016' => "Aldehuela de Liestos",
			'50017' => "Alfajarín",
			'50018' => "Alfamén",
			'50019' => "Alforque",
			'50020' => "Alhama de Aragón",
			'50021' => "Almochuel",
			'50022' => "Almolda, La",
			'50023' => "Almonacid de la Cuba",
			'50024' => "Almonacid de la Sierra",
			'50025' => "Almunia de Doña Godina, La",
			'50026' => "Alpartir",
			'50027' => "Ambel",
			'50028' => "Anento",
			'50029' => "Aniñón",
			'50030' => "Añón de Moncayo",
			'50031' => "Aranda de Moncayo",
			'50032' => "Arándiga",
			'50033' => "Ardisa",
			'50034' => "Ariza",
			'50035' => "Artieda",
			'50036' => "Asín",
			'50037' => "Atea",
			'50038' => "Ateca",
			'50039' => "Azuara",
			'50040' => "Badules",
			'50041' => "Bagüés",
			'50042' => "Balconchán",
			'50043' => "Bárboles",
			'50044' => "Bardallur",
			'50045' => "Belchite",
			'50046' => "Belmonte de Gracián",
			'50047' => "Berdejo",
			'50048' => "Berrueco",
			'50901' => "Biel",
			'50050' => "Bijuesca",
			'50051' => "Biota",
			'50052' => "Bisimbre",
			'50053' => "Boquiñeni",
			'50054' => "Bordalba",
			'50055' => "Borja",
			'50056' => "Botorrita",
			'50057' => "Brea de Aragón",
			'50058' => "Bubierca",
			'50059' => "Bujaraloz",
			'50060' => "Bulbuente",
			'50061' => "Bureta",
			'50062' => "Burgo de Ebro, El",
			'50063' => "Buste, El",
			'50064' => "Cabañas de Ebro",
			'50065' => "Cabolafuente",
			'50066' => "Cadrete",
			'50067' => "Calatayud",
			'50068' => "Calatorao",
			'50069' => "Calcena",
			'50070' => "Calmarza",
			'50071' => "Campillo de Aragón",
			'50072' => "Carenas",
			'50073' => "Cariñena",
			'50074' => "Caspe",
			'50075' => "Castejón de Alarba",
			'50076' => "Castejón de las Armas",
			'50077' => "Castejón de Valdejasa",
			'50078' => "Castiliscar",
			'50079' => "Cervera de la Cañada",
			'50080' => "Cerveruela",
			'50081' => "Cetina",
			'50092' => "Chiprana",
			'50093' => "Chodes",
			'50082' => "Cimballa",
			'50083' => "Cinco Olivas",
			'50084' => "Clarés de Ribota",
			'50085' => "Codo",
			'50086' => "Codos",
			'50087' => "Contamina",
			'50088' => "Cosuenda",
			'50089' => "Cuarte de Huerva",
			'50090' => "Cubel",
			'50091' => "Cuerlas, Las",
			'50094' => "Daroca",
			'50095' => "Ejea de los Caballeros",
			'50096' => "Embid de Ariza",
			'50098' => "Encinacorba",
			'50099' => "Épila",
			'50100' => "Erla",
			'50101' => "Escatrón",
			'50102' => "Fabara",
			'50104' => "Farlete",
			'50105' => "Fayón",
			'50106' => "Fayos, Los",
			'50107' => "Figueruelas",
			'50108' => "Fombuena",
			'50109' => "Frago, El",
			'50110' => "Frasno, El",
			'50111' => "Fréscano",
			'50113' => "Fuendejalón",
			'50114' => "Fuendetodos",
			'50115' => "Fuentes de Ebro",
			'50116' => "Fuentes de Jiloca",
			'50117' => "Gallocanta",
			'50118' => "Gallur",
			'50119' => "Gelsa",
			'50120' => "Godojos",
			'50121' => "Gotor",
			'50122' => "Grisel",
			'50123' => "Grisén",
			'50124' => "Herrera de los Navarros",
			'50125' => "Ibdes",
			'50126' => "Illueca",
			'50128' => "Isuerre",
			'50129' => "Jaraba",
			'50130' => "Jarque de Moncayo",
			'50131' => "Jaulín",
			'50132' => "Joyosa, La",
			'50133' => "Lagata",
			'50134' => "Langa del Castillo",
			'50135' => "Layana",
			'50136' => "Lécera",
			'50138' => "Lechón",
			'50137' => "Leciñena",
			'50139' => "Letux",
			'50140' => "Litago",
			'50141' => "Lituénigo",
			'50142' => "Lobera de Onsella",
			'50143' => "Longares",
			'50144' => "Longás",
			'50146' => "Lucena de Jalón",
			'50147' => "Luceni",
			'50148' => "Luesia",
			'50149' => "Luesma",
			'50150' => "Lumpiaque",
			'50151' => "Luna",
			'50152' => "Maella",
			'50153' => "Magallón",
			'50154' => "Mainar",
			'50155' => "Malanquilla",
			'50156' => "Maleján",
			'50160' => "Mallén",
			'50157' => "Malón",
			'50159' => "Maluenda",
			'50161' => "Manchones",
			'50162' => "Mara",
			'50163' => "María de Huerva",
			'50902' => "Marracos",
			'50164' => "Mediana de Aragón",
			'50165' => "Mequinenza",
			'50166' => "Mesones de Isuela",
			'50167' => "Mezalocha",
			'50168' => "Mianos",
			'50169' => "Miedes de Aragón",
			'50170' => "Monegrillo",
			'50171' => "Moneva",
			'50172' => "Monreal de Ariza",
			'50173' => "Monterde",
			'50174' => "Montón",
			'50175' => "Morata de Jalón",
			'50176' => "Morata de Jiloca",
			'50177' => "Morés",
			'50178' => "Moros",
			'50179' => "Moyuela",
			'50180' => "Mozota",
			'50181' => "Muel",
			'50182' => "Muela, La",
			'50183' => "Munébrega",
			'50184' => "Murero",
			'50185' => "Murillo de Gállego",
			'50186' => "Navardún",
			'50187' => "Nigüella",
			'50188' => "Nombrevilla",
			'50189' => "Nonaspe",
			'50190' => "Novallas",
			'50191' => "Novillas",
			'50192' => "Nuévalos",
			'50193' => "Nuez de Ebro",
			'50194' => "Olvés",
			'50195' => "Orcajo",
			'50196' => "Orera",
			'50197' => "Orés",
			'50198' => "Oseja",
			'50199' => "Osera de Ebro",
			'50200' => "Paniza",
			'50201' => "Paracuellos de Jiloca",
			'50202' => "Paracuellos de la Ribera",
			'50203' => "Pastriz",
			'50204' => "Pedrola",
			'50205' => "Pedrosas, Las",
			'50206' => "Perdiguera",
			'50207' => "Piedratajada",
			'50208' => "Pina de Ebro",
			'50209' => "Pinseque",
			'50210' => "Pintanos, Los",
			'50211' => "Plasencia de Jalón",
			'50212' => "Pleitas",
			'50213' => "Plenas",
			'50214' => "Pomer",
			'50215' => "Pozuel de Ariza",
			'50216' => "Pozuelo de Aragón",
			'50217' => "Pradilla de Ebro",
			'50218' => "Puebla de Albortón",
			'50219' => "Puebla de Alfindén, La",
			'50220' => "Puendeluna",
			'50221' => "Purujosa",
			'50222' => "Quinto",
			'50223' => "Remolinos",
			'50224' => "Retascón",
			'50225' => "Ricla",
			'50227' => "Romanos",
			'50228' => "Rueda de Jalón",
			'50229' => "Ruesca",
			'50241' => "Sabiñán",
			'50230' => "Sádaba",
			'50231' => "Salillas de Jalón",
			'50232' => "Salvatierra de Esca",
			'50233' => "Samper del Salz",
			'50234' => "San Martín de la Virgen de Moncayo",
			'50235' => "San Mateo de Gállego",
			'50236' => "Santa Cruz de Grío",
			'50237' => "Santa Cruz de Moncayo",
			'50238' => "Santa Eulalia de Gállego",
			'50239' => "Santed",
			'50240' => "Sástago",
			'50242' => "Sediles",
			'50243' => "Sestrica",
			'50244' => "Sierra de Luna",
			'50245' => "Sigüés",
			'50246' => "Sisamón",
			'50247' => "Sobradiel",
			'50248' => "Sos del Rey Católico",
			'50249' => "Tabuenca",
			'50250' => "Talamantes",
			'50251' => "Tarazona",
			'50252' => "Tauste",
			'50253' => "Terrer",
			'50254' => "Tierga",
			'50255' => "Tobed",
			'50256' => "Torralba de los Frailes",
			'50257' => "Torralba de Ribota",
			'50258' => "Torralbilla",
			'50259' => "Torrehermosa",
			'50260' => "Torrelapaja",
			'50261' => "Torrellas",
			'50262' => "Torres de Berrellén",
			'50263' => "Torrijo de la Cañada",
			'50264' => "Tosos",
			'50265' => "Trasmoz",
			'50266' => "Trasobares",
			'50267' => "Uncastillo",
			'50268' => "Undués de Lerda",
			'50269' => "Urrea de Jalón",
			'50270' => "Urriés",
			'50271' => "Used",
			'50272' => "Utebo",
			'50274' => "Val de San Martín",
			'50273' => "Valdehorna",
			'50275' => "Valmadrid",
			'50276' => "Valpalmas",
			'50277' => "Valtorres",
			'50278' => "Velilla de Ebro",
			'50279' => "Velilla de Jiloca",
			'50280' => "Vera de Moncayo",
			'50281' => "Vierlas",
			'50283' => "Villadoz",
			'50284' => "Villafeliche",
			'50285' => "Villafranca de Ebro",
			'50286' => "Villalba de Perejil",
			'50287' => "Villalengua",
			'50903' => "Villamayor de Gállego",
			'50288' => "Villanueva de Gállego",
			'50290' => "Villanueva de Huerva",
			'50289' => "Villanueva de Jiloca",
			'50291' => "Villar de los Navarros",
			'50292' => "Villarreal de Huerva",
			'50293' => "Villarroya de la Sierra",
			'50294' => "Villarroya del Campo",
			'50282' => "Vilueña, La",
			'50295' => "Vistabella",
			'50296' => "Zaida, La",
			'50297' => "Zaragoza",
			'50298' => "Zuera",
			'51001' => "Ceuta",
			'52001' => "Melilla",
		];
	}
}
